const cp = require('child_process');
const os = require('os');
const path = require('path');

class Ssh {
  constructor({ host, options = {} }) {
    this._host = host;
    this._options = {
      cwd: process.cwd(),
      ...options,
      env: { ...process.env, ...options.env },
    };
    this._ctl = path.join(os.tmpdir(), `ctl-${this._host}`);
  }

  async run(args, options = {}) {
    const commandArgs = ['-S', this._ctl, ...args, this._host];
    const spawnOptions = {
      ...this._options,
      ...options,
      env: { ...this._options.env, ...options.env },
      stdio: ['ignore', 'pipe', 'pipe'],
    };

    let stderr = '';
    let stdout = '';

    return new Promise((resolve, reject) => {
      const proc = cp.spawn('ssh', commandArgs, spawnOptions);

      proc.stdout.setEncoding('utf8');
      proc.stdout.on('data', (data) => (stdout += data.toString()));

      proc.stderr.setEncoding('utf8');
      proc.stderr.on('data', (data) => (stderr += data.toString()));

      proc.on('error', (e) => {
        console.error(e);
        reject(e);
      });

      proc.on('exit', (code, signal) => {
        resolve({
          code,
          signal,
          stdout: stdout.replace(/\n$/, ''),
          stderr: stderr.replace(/\n$/, ''),
        });
      });
    });
  }

  async connect() {
    const res = await this.run(['-M', '-fN']);

    return res;
  }

  async forward(ports) {
    if (ports.length === 0) {
      throw new Error('port forwardings is not set');
    }

    const args = ['-O', 'forward'];

    for (const { port, host, hostPort } of ports) {
      args.push(`-L ${port}:${host}:${hostPort}`);
    }

    const res = await this.run(args);

    return res;
  }

  async check() {
    const { code } = await this.run(['-O', 'check'], { stdio: 'ignore' });

    return code === 0;
  }

  async disconnect() {
    const result = await this.run(['-O', 'exit'], { stdio: 'ignore' });

    return result;
  }
}

module.exports = { Ssh };
