#!/usr/bin/env node
const readline = require('readline');
const { stdin: input, stdout: output } = require('process');
const axios = require('axios').default;

const { getItemsToRemove } = require('./lib/keyset-helpers');
const getConfig = require('./config');

const {
  tanker: { project, branch, default: defaultLang },
} = getConfig();

const TANKER_HOST = 'https://tanker.yandex-team.ru';
const getExportKeysTankerUrl = () =>
  `${TANKER_HOST}/api/v1/project/${project}/branch/${branch}/export?sort=name&languages=${defaultLang}`;
const getDeleteKeysTankerUrl = (keyset) =>
  `${TANKER_HOST}/api/v1/project/${project}/keyset/${keyset}/branch/${branch}/delete`;
const getPatchKeysetTankerUrl = (keyset) =>
  `${TANKER_HOST}/api/v1/project/${project}/keyset/${keyset}`;

const authHeaders = {
  Authorization: `OAuth ${process.env.TANKER_TOKEN || process.env.TANKER_API_TOKEN}`,
};

const rl = readline.createInterface({ input, output });

/**
 *
 * @param {string} prompt
 * @returns {string}
 */
async function question(prompt) {
  return new Promise((resolve) => {
    rl.question(prompt, resolve);
  });
}

async function deleteKeysFromKeyset(keyset, keys) {
  const url = getDeleteKeysTankerUrl(keyset);
  const data = { commit_message: `Cleaning by ${process.env.USER}`, keys: [].concat(keys) };

  console.log(`Удаляем ключи из ${keyset}: ${keys.map((key) => `"${key}"`).join(', ')}`);

  await axios.post(url, data, {
    headers: authHeaders,
  });
}

async function deleteKeyset(keyset) {
  const url = getPatchKeysetTankerUrl(keyset);
  const data = { branch, status: 'ARCHIVED' };

  console.log(`Удаляем кисет ${keyset}`);

  await axios.patch(url, data, {
    headers: authHeaders,
  });
}

async function main() {
  const fullTankerKeys = (
    await axios.get(getExportKeysTankerUrl(), {
      headers: authHeaders,
    })
  ).data.keysets;

  const keysetsInTanker = {};

  Object.entries(fullTankerKeys).forEach(([keysetName, { keys }]) => {
    Object.keys(keys).forEach((key) => {
      (keysetsInTanker[keysetName] || (keysetsInTanker[keysetName] = {}))[key] = true;
    });
  });

  const orphanedData = require('./stale.json');
  const toRemoveData = getItemsToRemove(orphanedData, keysetsInTanker);

  if (toRemoveData.keysToRemove + toRemoveData.keysetsToRemove === 0) {
    console.log('Очищать нечего');

    return;
  }

  let msg = 'Будут удалены\n';

  if (toRemoveData.keysToRemove) {
    msg += `\n * Неиспользуемых ключей: ${toRemoveData.keysToRemove}\n`;
    msg += toRemoveData.list
      .filter((item) => item.type === 'keys')
      .map(({ keyset, keys }) => keys.map((key) => `   * ${keyset} / ${key}\n`))
      .flat()
      .join('');
  }

  if (toRemoveData.keysetsToRemove) {
    msg += `\n * Пустых кисетов: ${toRemoveData.keysetsToRemove}\n`;
    msg += toRemoveData.list
      .filter((item) => item.type === 'keyset')
      .map((item) => `   * ${item.keyset}\n`)
      .join('');
  }

  console.log(msg);

  const ans = await question('Хотите продолжить (y/N)? ');

  rl.close();

  if (ans.toLowerCase() !== 'y') {
    return;
  }

  for (const item of toRemoveData.list) {
    if (item.type === 'keys') {
      await deleteKeysFromKeyset(item.keyset, item.keys);
    }

    if (item.type === 'keyset') {
      await deleteKeyset(item.keyset);
    }
  }
}

main();
