const path = require('path');
const fs = require('fs');
const { sync: rmSync } = require('rimraf');
const assert = require('assert');
const _ = require('lodash');
const {
  getKeyEntries,
  getKeysetToPathMap,
  getItemsToRemove,
  getKeysetsOnFS,
} = require('./keyset-helpers');

const renderKeysetFile = _.template('export const ${ lang } = ${ data };\n');

module.exports = function dispatcher(arg) {
  try {
    const data = JSON.parse(arg.data);
    const entries = getKeyEntries(data);
    const keysetToPathMap = getKeysetToPathMap();
    const indexTemplate = fs.readFileSync(path.join(__dirname, 'index-template.ts'), 'utf-8');
    const result = {};
    const orphanedData = {};
    const keysetsOnFS = getKeysetsOnFS();

    // Находим, где должен лежать каждый будущий файл
    // Также добавляем необходимые индексные файлы
    entries.forEach((entry) => {
      let keysetDir;
      const keys = {};

      // перебираем все ключи в кейсете
      Object.entries(entry.keys).forEach(([key, translate]) => {
        const dir = keysetToPathMap[`${entry.keyset}/${key}`];

        // если в маппинге (после парсинга исходников на ФС) не нашлось записи для этого ключа,
        // то он устарел и его нужно удалить
        if (!dir) {
          const orphanedKeyset = orphanedData[entry.keyset] || (orphanedData[entry.keyset] = {});

          (orphanedKeyset[key] || (orphanedKeyset[key] = {}))[entry.lang] = translate;

          return;
        }

        if (!keysetDir) {
          keysetDir = dir;
        } else {
          // в маппинге пути для всех ключей одного кейсета должны совпадать
          assert.equal(dir, keysetDir, 'Обнаружены разные пути для ключей одного кисета');
        }

        // ключ не устаревший и с правильной директорией заносим в список для экспорта в кейсеты на ФС
        keys[key] = translate;
      });

      if (!Object.keys(keys).length) {
        return;
      }

      assert.ok(keysetDir, `Пустая директория для кисета ${keys}`);

      keysetsOnFS.delete(keysetDir);

      const filePath = path.join(keysetDir, `${entry.lang}.ts`);
      const keysetName = path.basename(path.dirname(keysetDir));
      const indexFile = path.join(keysetDir, 'index.ts');

      result[filePath] = renderKeysetFile({
        lang: entry.lang,
        data: JSON.stringify(keys, null, 4),
      });

      // если для данного кейсета/каталога еще нет индексного файла, то добавляем его
      if (!result[indexFile]) {
        result[indexFile] = keysetName
          ? indexTemplate.replace('lang-[request]', `lang-${keysetName}-[request]`)
          : indexTemplate;
      }
    });

    // eslint-disable-next-line array-callback-return
    keysetsOnFS.forEach((key) => /\.i18n$/.test(key) && rmSync(key));

    if (Object.keys(orphanedData).length > 0) {
      console.warn(
        'Были обнаружены потерянные ключи. Полный список сохранен в файл ' +
          path.resolve(__dirname, '../stale.json'),
      );

      const healthStatus = getItemsToRemove(orphanedData, data.ru);

      if (healthStatus.keysToRemove + healthStatus.keysetsToRemove) {
        let msg = 'Было найдено\n';

        healthStatus.keysToRemove &&
          (msg += ` * ${healthStatus.keysToRemove} неиспользуемых ключей\n`);
        healthStatus.keysetsToRemove &&
          (msg += ` * ${healthStatus.keysetsToRemove} пустых кисетов\n`);

        msg +=
          ' доступных для удаления\nЗапустите команду ./.tanker/clean.js для очистки лишних ключей из танкера\n';
        console.log(msg);
      }

      console.log(`Всего рабочих ключей: ${healthStatus.normalKeys}\n`);
    }

    fs.writeFileSync(
      path.resolve(__dirname, '../stale.json'),
      JSON.stringify(orphanedData, null, 4),
      'utf-8',
    );

    return result;
  } catch (e) {
    // Танкер имеет свойство проглатывать ошибки и молча вываливаться, поэтому явно выводим в консоль ошибку
    // и бросаем её дальше, чтобы сработал обычный обработчик
    console.error('Error: ', e);

    throw e;
  }
};
