'use strict';

/**
 * Слегка модифицированный файл из @yandex-int/tanker-ts-i18n/lib/extract.js
 */
const path = require('path');
const ts = require('typescript');
const config = require('../config')();

const lineBreakRegex = /\r\n|\r|\n/;
const lineBreakRegexGlobal = /\r\n|\r|\n/g;

function getLines(input) {
  return input.split(lineBreakRegex);
}

function getLineInfo(code) {
  const lines = [];

  lineBreakRegexGlobal.lastIndex = 0;
  let lastOffset = 0;
  let match;
  let lineNumber = 0;

  while ((match = lineBreakRegexGlobal.exec(code)) !== null) {
    lines.push({
      number: ++lineNumber,
      offset: lastOffset,
      text: code.substring(lastOffset, match.index),
      lineBreak: match[0],
    });
    lastOffset = match.index + match[0].length;
  }
  lines.push({
    number: ++lineNumber,
    offset: lastOffset,
    text: code.substr(lastOffset),
    lineBreak: null,
  });

  return lines;
}

function getNodeRange(node) {
  if (typeof node.getStart === 'function' && typeof node.getEnd === 'function') {
    return [node.getStart(), node.getEnd()];
  } else if (typeof node.pos !== 'undefined' && typeof node.end !== 'undefined') {
    return [node.pos, node.end];
  }

  return [0, 0];
}

function extractKeyFromCall(callExpression) {
  let key;

  if (callExpression.arguments.length) {
    const firstArg = callExpression.arguments[0];

    if (
      firstArg.kind === ts.SyntaxKind.StringLiteral ||
      firstArg.kind === ts.SyntaxKind.NoSubstitutionTemplateLiteral
    ) {
      const keyLiteral = firstArg;

      key = {
        name: keyLiteral.text,
        nameRaw: keyLiteral.getText(),
        range: getNodeRange(callExpression),
        location: { line: 0, column: 0 },
        fragment: '',
        keyset: '',
      };
    }
    const secondArg = callExpression.arguments[1];

    if (secondArg && secondArg.kind === ts.SyntaxKind.ObjectLiteralExpression) {
      const optionsExpression = secondArg;

      optionsExpression.properties.forEach(function (prop) {
        let _a;

        if (ts.isPropertyAssignment(prop)) {
          const assignment = prop;
          const initializerText =
            prop.initializer.kind === ts.SyntaxKind.StringLiteral
              ? prop.initializer.text
              : prop.initializer.getText();
          const paramName = assignment.name.getText();

          if (key) {
            if (paramName === 'context') {
              key.context = initializerText;
            } else {
              key.params || (key.params = []);
              key.params.push(((_a = {}), (_a[paramName] = initializerText), _a));
            }
          }
        }
      });
    }
  }

  return key;
}

function isI18nCall(callExpression, i18nIdentifierName) {
  return callExpression.expression.getText() === i18nIdentifierName;
}

function getRequiredI18nIdentifierName(requiredVarDecl, i18nModuleName) {
  if (
    requiredVarDecl.initializer &&
    requiredVarDecl.initializer.kind === ts.SyntaxKind.CallExpression
  ) {
    const callExpression = requiredVarDecl.initializer;

    if (
      callExpression.expression.getText() === 'require' &&
      callExpression.arguments.length === 1 &&
      callExpression.arguments[0].getText() === "'" + i18nModuleName + "'"
    ) {
      return requiredVarDecl.name.getText();
    }
  }

  return '';
}

function getLocationOfFragment(sourceCode, fragment) {
  const linesInfo = getLineInfo(sourceCode);
  const fragmentLines = getLines(fragment);
  const fragmentStart = sourceCode.indexOf(fragment);

  for (let i = 0; ~fragmentStart && i < linesInfo.length - 1; i++) {
    const info = linesInfo[i];

    if (info.offset === fragmentStart) {
      return { line: info.number, column: 0 };
    }
    if (info.offset > fragmentStart) {
      const line = info.number - 1;
      const prevLineInfo = linesInfo[i - 1];
      const column = prevLineInfo.text.indexOf(fragmentLines[0]);

      return { line: line, column: column };
    }
  }

  return { line: 0, column: 0 };
}

function getKeySetName(importDecl, i18nKeysetExtension, srcFileName) {
  const moduleNameWithQuotes = importDecl.moduleSpecifier.getText();
  const moduleName = moduleNameWithQuotes.substring(1, moduleNameWithQuotes.length - 1);
  const reg = new RegExp('/' + i18nKeysetExtension + '$');

  if (reg.test(moduleName)) {
    let keysetName = path.resolve(path.dirname(srcFileName));

    config.levels.every(
      (level, idx) =>
        keysetName === (keysetName = keysetName.replace(level, config.levelPrefixes[idx]))
    );

    if (keysetName.charAt(0) === '/') {
      keysetName = keysetName.substring(1);
    }

    return keysetName.replace(/\//g, '_');
  }

  return '';
}

function getImportI18nIdentifierName(importDecl, i18nModuleName, keyName) {
  const moduleNameWithQuotes = importDecl.moduleSpecifier.getText();
  const moduleName = moduleNameWithQuotes.substring(1, moduleNameWithQuotes.length - 1);
  const reg = new RegExp(i18nModuleName + '$');

  if (reg.test(moduleName) && importDecl.importClause) {
    let identifier = undefined;

    if (keyName === 'default') {
      identifier = importDecl.importClause.name || importDecl.importClause.namedBindings.name;
    } else {
      const namedBindings =
        importDecl.importClause.namedBindings && importDecl.importClause.namedBindings.elements;

      if (namedBindings && namedBindings.length) {
        identifier = namedBindings.find(function (item) {
          const name = (item.propertyName && item.propertyName.text) || item.name.text;

          return name === keyName;
        });

        identifier = identifier && identifier.name;
      }
    }

    return identifier ? identifier.text : '';
  }

  return '';
}

function extractI18NKeysFromSourceFile(
  sourceFile,
  i18nModuleName,
  i18nKeysetExtension,
  srcFileName,
) {
  if (i18nModuleName === undefined) {
    i18nModuleName = 'bem-i18n';
  }
  if (i18nKeysetExtension === undefined) {
    i18nKeysetExtension = 'i18n';
  }
  const keys = [];
  let keySetName = '';
  let i18nIdentifierName;
  let currentKeyName;

  function next(node) {
    node.forEachChild(walk);
  }
  function walk(node) {
    const firstLevel = node.parent && node.parent.kind === ts.SyntaxKind.SourceFile;

    if (firstLevel && node.kind === ts.SyntaxKind.ImportDeclaration) {
      !i18nIdentifierName &&
        (i18nIdentifierName = getImportI18nIdentifierName(node, i18nModuleName, currentKeyName));
      !keySetName && (keySetName = getKeySetName(node, i18nKeysetExtension, srcFileName));
    }
    if (node.kind === ts.SyntaxKind.VariableDeclaration) {
      !i18nIdentifierName &&
        (i18nIdentifierName = getRequiredI18nIdentifierName(node, i18nModuleName));
    }
    if (node.kind === ts.SyntaxKind.CallExpression) {
      const callExpression = node;

      if (i18nIdentifierName && isI18nCall(callExpression, i18nIdentifierName)) {
        const parent = callExpression.parent;

        if (!parent) {
          return next(node);
        }

        const i18nKey = extractKeyFromCall(callExpression);

        i18nKey && keys.push(i18nKey);

        return next(node);
      }
    }

    return next(node);
  }
  ts.bindSourceFile(sourceFile, {
    target: ts.ScriptTarget.ES5,
    module: ts.ModuleKind.CommonJS,
  });
  const text = sourceFile.text;

  ['i18nRaw', 'i18n'].forEach(function (keyName) {
    currentKeyName = keyName;
    i18nIdentifierName = '';
    ts.forEachChild(sourceFile, walk);
  });
  keys.forEach(function (key) {
    key.fragment = text.substring(key.range[0], key.range[1]);
    key.location = getLocationOfFragment(text, key.fragment);
    key.keyset = keySetName || '';
    (key.params && key.params.length) || (key.params = undefined);
  });

  return keys;
}

function extractI18NKeysFromSourceCode(
  sourceCode,
  i18nModuleName,
  i18nKeysetExtension,
  sourceFilePath,
) {
  if (sourceFilePath === undefined) {
    sourceFilePath = 'sourceCode.tsx';
  }
  const sourceFileName = path.basename(sourceFilePath);
  const sourceFile = ts.createSourceFile(sourceFileName, sourceCode, ts.ScriptTarget.ES5, true);

  return extractI18NKeysFromSourceFile(
    sourceFile,
    i18nModuleName,
    i18nKeysetExtension,
    sourceFilePath,
  );
}
exports.extractI18NKeysFromSourceCode = extractI18NKeysFromSourceCode;
