const path = require('path');
const { execSync } = require('child_process');
const assert = require('assert');
const _ = require('lodash');

const config = require('../config');
const langs = require('../../src/shared/i18n/common').langs;

/**
 * Из структуры: lang -> keyset -> key: value делаем плоский список будущих файлов
 *
 * @param {Object.<string, Object.<string, Object.<string, string>>>} data - данные из танкера
 * @returns {{keys: Object, lang: String, keyset: String}[]}
 */
function getKeyEntries(data) {
  const entries = [];

  Object.entries(data).forEach(([lang, keysets]) => {
    Object.entries(keysets).forEach(([keysetName, keysetData]) => {
      entries.push({
        lang: lang,
        keyset: keysetName,
        keys: Object.keys(keysetData).reduce((keys, key) => {
          const translation = keysetData[key];

          keys[key] = Array.isArray(translation)
            ? {
                one: translation[0],
                some: translation[1],
                many: translation[2],
                none: translation[3],
              }
            : translation;

          return keys;
        }, {}),
      });
    });
  });

  return entries;
}

/**
 * Из файла map.json получаем маппинг для директорий, где лежат соответствующие кисеты
 *
 * @returns {Object.<string, string>}
 */
function getKeysetToPathMap() {
  const keysetToPathMap = {};

  const { root } = config();

  // eslint-disable-next-line array-callback-return
  Object.values(require('../map.json')).forEach((items) =>
    items.forEach((mapEntry) => {
      const keyset = `${mapEntry.keyset}/${mapEntry.key}`;
      // путь по которому должен быть расположен кейсет из маппинга
      const dir = path.resolve(root, path.join(path.dirname(mapEntry.path), 'i18n'));

      assert.ok(
        !keysetToPathMap[keyset] || dir === keysetToPathMap[keyset],
        'Обнаружены разные пути для одинаковых кисетов',
      );
      keysetToPathMap[keyset] = dir;
    }),
  );

  return keysetToPathMap;
}

/**
 * Вычисляет данные по доступным к удалению ключам танкера
 * Из списка неиспользуемых ключей отфильтровывает ключи без перевода
 * Также среди всех кейсетов ищет пустые (в том числе и после удаления лишних ключей)
 *
 * @param {Record<string, Record<string, string>>} orphanedData - хеш с неиспользованными ключами
 * @param {Record<string, Record<string, string>>} keysetsInTankerOrig - хеш с полным списком ключей
 */
function getItemsToRemove(orphanedData, keysetsInTankerOrig) {
  const toRemove = [];
  const keysetsInTanker = _.cloneDeep(keysetsInTankerOrig);
  let keys = 0;
  let keysets = 0;

  Object.entries(orphanedData).forEach(([keyset, keysSet]) => {
    let keysToRemove;

    Object.entries(keysSet).forEach(([key, translations]) => {
      const original = translations.ru;
      const haveTranslation = config().removeTranslatedKeys ? false : langs.some(
          (lang) => translations[lang] && (translations[lang] !== original),
        );

      if (!haveTranslation) {
        (keysToRemove || (keysToRemove = [])).push(key);
        keys++;

        try {
          delete keysetsInTanker[keyset][key];
        } catch (e) {
          console.log(
            `Обнаружена неконсистентность данных для ключа ${keyset}/${key}`,
            keysetsInTanker[keyset],
            '\nВозможно ваша локальная копия исходных кодов устарела',
          );
          process.exit(1);
        }
      }
    });

    if (keysToRemove) {
      toRemove.push({ keyset, keys: keysToRemove, type: 'keys' });
    }
  });

  Object.entries(keysetsInTanker).forEach(([keyset, keys]) => {
    if (Object.keys(keys).length === 0) {
      keysets++;
      toRemove.push({ keyset, type: 'keyset' });
    }
  });

  const normalKeys = Object.values(keysetsInTanker).reduce(
    (sum, keys) => sum + Object.keys(keys).length,
    0,
  );

  return {
    keysToRemove: keys,
    keysetsToRemove: keysets,
    normalKeys,
    list: toRemove,
  };
}

function getKeysetsOnFS() {
  const { levels, root } = config();
  const rawFileList = execSync(`find ${levels.join(' ')} -type f -path "*/i18n/ru.ts"`, {
    cwd: root,
    encoding: 'utf-8',
    timeout: 30000,
  });

  const totalKeysets = new Set(
    rawFileList
      .split('\n')
      .map((line) => line.trim())
      .filter(Boolean)
      .map((filename) => path.dirname(filename)),
  );

  return totalKeysets;
}

module.exports = {
  getKeyEntries,
  getKeysetToPathMap,
  getItemsToRemove,
  getKeysetsOnFS,
};
