declare namespace ExpressHttpLangdetect {
  interface AvailableLanguagesObject {
    [tld: string]: string | string[];
  }

  type AvailableLanguagesFunction = (req: Request) => string | string[] | AvailableLanguagesObject;

  interface Language {
    id: string;
    name: string;
  }
}

declare namespace Express {
  interface Request {
    /**
     * Detected user language by express-http-langdetect middleware
     *
     * See:
     * https://github.yandex-team.ru/toolbox/express-http-langdetect
     */
    langdetect: ExpressHttpLangdetect.Language;
  }
}

declare module '@yandex-int/express-http-langdetect' {
  import { RequestHandler } from 'express';
  import { GotOptions } from 'got';

  export interface ExpressHttpLangdetectOptions {
    clientOptions?: GotOptions<string | null>;

    /** Default service language */
    defaultLanguage?: string;

    /**
     * This language is returned in case of errors if defaultLanguage is not set.
     * Defaults to {id: "ru", name: "Ru"}
     *
     * Set to false if you don't need any fallback value at all
     */
    fallbackLanguage?: boolean | ExpressHttpLangdetect.Language;

    /** Add list of user-relevant languages to req.langdetect.list object */
    list?: boolean;

    /**
     * Available languages can be one of the following:
     *   1. String, that contains all language codes (ids) split with ',' sign (i.e. 'en,ru,tr')
     *
     *   2. Array, that contains language codes, that is joined using ',' to get a string
     *      described above
     *
     *   3. Object, that has service TLDs as keys and arrays or strings in format
     *      described above
     *
     *   4. Function, that accepts express request object and should return any of the above.
     */
    availableLanguages?:
      | string
      | string[]
      | ExpressHttpLangdetect.AvailableLanguagesObject
      | ExpressHttpLangdetect.AvailableLanguagesFunction;

    /** API endpoint (http://langdetect.qloud.yandex.ru by default) */
    server?: string;
  }

  /**
   * Detects user language into req.langdetect
   *
   * See:
   * https://github.yandex-team.ru/toolbox/express-http-langdetect
   */
  function expressHttpLangdetect(options?: ExpressHttpLangdetectOptions): RequestHandler;

  export default expressHttpLangdetect;
}
