import { VFC, useCallback } from 'react';

import bubbleImageSrc from '@client/shared/assets/bubble.png';
import {
  Button,
  CardTemplate,
  CodeField,
  Form,
  ImageIcon,
  Text,
  Timer,
} from '@client/shared/ui-kit';

import { i18n } from './i18n';
import { SmsCodeConfirmationEvent, SmsCodeConfirmationEventType } from './types';

import styles from './SmsCodeConfirmationForm.module.css';

export interface SmsCodeConfirmationFormProps {
  code?: string;
  number: string;
  expiryTimestamp?: number;
  error?: string;
  invalid?: boolean;
  pending?: boolean;
  onAction: (event: SmsCodeConfirmationEvent) => void;
}

export const SmsCodeConfirmationForm: VFC<SmsCodeConfirmationFormProps> = (props) => {
  const { code = '', number, expiryTimestamp, error, invalid, pending, onAction } = props;

  const handleChange = useCallback(
    (value: string) => {
      if (value === '' || /^[0-9]+$/.test(value)) {
        onAction({ type: SmsCodeConfirmationEventType.CHANGE_CODE, code: value });
      }
    },
    [onAction],
  );

  const handleSubmit = useCallback(() => {
    onAction({ type: SmsCodeConfirmationEventType.VERIFY_CODE, code });
  }, [onAction, code]);

  const handleResend = useCallback(() => {
    onAction({ type: SmsCodeConfirmationEventType.RESEND_CODE });
  }, [onAction]);

  const resendButtonText = i18n('Отправить новый код');

  return (
    <Form onSubmit={handleSubmit}>
      <CardTemplate>
        <CardTemplate.Media>
          <ImageIcon src={bubbleImageSrc} size={96} />
        </CardTemplate.Media>

        <CardTemplate.Content align="center">
          <Text variant="heading-m" weight="bold">
            {i18n('Введите код, отправленный на номер {number}', {
              number,
            })}
          </Text>
        </CardTemplate.Content>

        <div className={styles.control}>
          <CodeField
            type="text"
            value={code}
            onChange={handleChange}
            onComplete={handleSubmit}
            disabled={pending}
            invalid={invalid}
            inputMode="numeric"
          />

          {error && (
            <Text variant="text-m-long" color="negative">
              {error}
            </Text>
          )}
        </div>

        <CardTemplate.Actions>
          <Button
            variant="action"
            size="xl"
            disabled={pending || code.length === 0}
            onPress={handleSubmit}
          >
            {i18n('Готово')}
          </Button>

          <Timer expiryTimestamp={expiryTimestamp}>
            {({ milliseconds, text }) => (
              <Button
                variant="clear"
                size="xl"
                disabled={pending || milliseconds > 0}
                onPress={handleResend}
              >
                {milliseconds > 0 ? `${resendButtonText}: ${text}` : resendButtonText}
              </Button>
            )}
          </Timer>
        </CardTemplate.Actions>
      </CardTemplate>
    </Form>
  );
};
