import cx from 'classnames';
import { FC } from 'react';

import { useEnvContext } from '@client/shared/libs/env';
import { AppBar, Button, Form, Modal, Section } from '@client/shared/ui-kit';
import { createSlot, useSlots } from '@yandex-id/components/libs/slots';

import { FieldsSection } from './ui/FieldsSection';
import { MultiField } from './ui/MultiField';

import styles from './DocumentForm.module.css';

export interface DocumentFormProps {
  isVisible: boolean;
  isDisabled: boolean;
  onClose: () => void;
  onSubmit: () => void;
}

export interface TitleSlotProps {
  text: string;
}

const TitleSlot = createSlot<TitleSlotProps>('DocumentViewer.Title');
const PreviewSlot = createSlot('DocumentViewer.Preview');

const _DocumentForm: FC<DocumentFormProps> = (props) => {
  const { isVisible, isDisabled, onClose, onSubmit, ...restProps } = props;
  const slots = useSlots(restProps);
  const {
    ua: { isYandexApp, isSearchApp },
  } = useEnvContext();

  const title = slots.get(TitleSlot);
  const preview = slots.get(PreviewSlot);

  return (
    <Modal
      hasClose={!(isYandexApp && isSearchApp)}
      className={styles.root}
      visible={isVisible}
      keepMounted={false}
      onClose={onClose}
    >
      {isYandexApp && isSearchApp && title && <AppBar title={title.props.text} state="back" />}
      <Form className={styles.form}>
        <Section
          className={cx(styles.section, {
            [styles.section_ppPreview]: isYandexApp && isSearchApp,
          })}
        >
          {!(isYandexApp && isSearchApp) && title && (
            <Section.Title>{title.props.text}</Section.Title>
          )}
          {preview && preview.rendered}
        </Section>
        {slots.children}
        <Section className={styles.section}>
          <Button type="submit" size="l" variant="action" onPress={onSubmit} disabled={isDisabled}>
            Сохранить
          </Button>
        </Section>
      </Form>
    </Modal>
  );
};

export const DocumentForm = Object.assign(_DocumentForm, {
  Title: TitleSlot,
  Preview: PreviewSlot,
  Section: FieldsSection,
  MultiField: MultiField,
});
