import { VFC, useCallback, useMemo, useState } from 'react';

import { DocumentImage } from '@client/shared/api/graphql';
import { Carousel, CarouselItem, Spin } from '@client/shared/ui-kit';

import { Target } from './types';
import { AddButton } from './ui/AddButton';
import { DocumentImageViewer, Url } from './ui/DocumentImageViewer';
import { Image } from './ui/Image';

import styles from './DocumentImageList.module.css';

export interface DocumentImageListProps {
  images: DocumentImage[];
  target: Target;
  isImageViewerVisible: boolean;
  isUploading: boolean;
  onImageClick: () => void;
  onImageViewerClose: () => void;
  onUpload: (images: FileList | null) => void;
  onDelete: (id: string) => void;
  onShare?: (index: number) => void | null;
  isDownloadDisabled?: boolean;
}

export const DocumentImageList: VFC<DocumentImageListProps> = (props) => {
  const {
    images,
    target,
    isImageViewerVisible,
    isUploading,
    onImageClick,
    onImageViewerClose,
    onUpload,
    onDelete,
    onShare,
    isDownloadDisabled,
  } = props;

  const imagesInList = useMemo(() => {
    if (target === 'view') {
      return images.slice();
    }

    return images.slice().reverse();
  }, [images, target]);

  const [currentImage, setCurrentImage] = useState(0);

  const urls: Url[] = imagesInList.map((image) => ({ originalUrl: image.originalUrl }));

  const handleOnOpenDelete = useCallback(
    (index: number) => {
      const id = imagesInList[index].id;

      onDelete(id);
    },
    [imagesInList, onDelete],
  );

  return (
    <>
      {imagesInList.length === 0 && !isUploading && (
        <AddButton target={target} onUpload={onUpload} single />
      )}
      {(imagesInList.length > 0 || isUploading) && (
        <Carousel className={styles.root} data-target={target}>
          {target === 'upload' && (
            <>
              <CarouselItem>
                <AddButton target={target} onUpload={onUpload} />
              </CarouselItem>
              {isUploading && (
                <CarouselItem>
                  <div className={styles.spinner} data-target={target}>
                    <Spin view="default" progress position="center" size="m" />
                  </div>
                </CarouselItem>
              )}
            </>
          )}
          {imagesInList.map((image, index) => (
            <CarouselItem key={image.id}>
              <Image
                onPress={() => {
                  onImageClick();
                  setCurrentImage(index);
                }}
                size={target === 'view' ? 'm' : 's'}
                src={image.previewUrl || image.originalUrl}
              />
            </CarouselItem>
          ))}
          {target === 'view' && (
            <>
              {isUploading && (
                <CarouselItem>
                  <div className={styles.spinner} data-target={target}>
                    <Spin
                      view="default"
                      progress
                      position="center"
                      size={target === 'view' ? 'm' : 'xs'}
                    />
                  </div>
                </CarouselItem>
              )}
              <CarouselItem>
                <AddButton target={target} onUpload={onUpload} />
              </CarouselItem>
            </>
          )}
        </Carousel>
      )}
      <DocumentImageViewer
        isActive={isImageViewerVisible}
        urls={urls}
        startIndex={currentImage}
        onClose={onImageViewerClose}
        onOpenShare={onShare}
        onOpenDelete={handleOnOpenDelete}
        isDownloadDisabled={isDownloadDisabled}
      />
    </>
  );
};
