import cx from 'classnames';
import React, { FC, RefObject, VFC, useCallback, useMemo, useRef } from 'react';
import ImageGallery, { ReactImageGalleryItem } from 'react-image-gallery';

import { useDevice } from '@client/shared/hooks';
import {
  ArrowShortBack,
  ArrowShortForward,
  DownloadOutline,
  ShareArrowOutline,
  TrashOutline,
} from '@client/shared/icons';
import { useEnvContext } from '@client/shared/libs/env';
import { AppBar, Button, ImagePresenter } from '@client/shared/ui-kit';

import { download } from './libs/download';

import styles from './DocumentImageViewer.module.css';
import 'react-image-gallery/styles/css/image-gallery.css';

export interface Url {
  originalUrl: string;
  previewUrl?: string;
}

interface DocumentImageViewerProps {
  isActive?: boolean;
  urls: Url[];
  onClose: () => void;
  onOpenDelete?: (index: number) => void;
  onOpenShare?: (index: number) => void;
  startIndex?: number;
  isDownloadDisabled?: boolean;
}

interface NavProps {
  galleryRef: RefObject<ImageGallery>;
}

const NavLeft: FC<NavProps> = (props) => {
  const { isTouch } = useDevice();

  if (isTouch) {
    return null;
  }

  const { galleryRef } = props;

  return (
    <Button
      className={cx(styles.nav, styles.nav_left)}
      onPress={() => {
        // @ts-expect-error в типах нет функции slideLeft, но она есть ._.
        galleryRef.current?.slideLeft();
      }}
    >
      <ArrowShortBack size={24} />
    </Button>
  );
};

const NavRight: FC<NavProps> = (props) => {
  const { isTouch } = useDevice();

  if (isTouch) {
    return null;
  }

  const { galleryRef } = props;

  return (
    <Button
      className={cx(styles.nav, styles.nav_right)}
      onPress={() => {
        // @ts-expect-error в типах нет функции slideRight, но она есть ._.
        galleryRef.current?.slideRight();
      }}
    >
      <ArrowShortForward size={24} />
    </Button>
  );
};

function getGalleryItems(urls: Url[]): ReactImageGalleryItem[] {
  return urls.map((url) => {
    return {
      originalClass: styles.image,
      original: url.originalUrl,
    };
  });
}

export const DocumentImageViewer: VFC<DocumentImageViewerProps> = (props) => {
  const galleryRef = useRef<ImageGallery>(null);
  const {
    isActive = false,
    onOpenDelete,
    onOpenShare,
    urls,
    onClose,
    startIndex,
    isDownloadDisabled,
  } = props;

  const onImageDelete = useCallback(() => {
    if (galleryRef.current && onOpenDelete) {
      const index = galleryRef.current.getCurrentIndex();

      onOpenDelete(index);
    }
  }, [onOpenDelete]);

  const onImageShare = useCallback(() => {
    if (galleryRef.current && onOpenShare) {
      const index = galleryRef.current.getCurrentIndex();

      onOpenShare(index);
    }
  }, [onOpenShare]);

  const onDownload = useCallback(() => {
    if (galleryRef.current) {
      const index = galleryRef.current.getCurrentIndex();
      const url = urls[index];
      const image = url.originalUrl;

      if (!image) {
        return;
      }

      download(image);
    }
  }, [urls]);

  const galleryItems = useMemo(() => getGalleryItems(urls), [urls]);

  const {
    ua: { isYandexApp, isSearchApp },
  } = useEnvContext();

  const searchAppTitle = isYandexApp && isSearchApp ? 'Документы' : null;

  const controls = (
    <div
      className={cx(styles.controls, {
        [styles.controls_hasShare]: onOpenShare,
        [styles.controls_wideButton]: onOpenDelete && isDownloadDisabled && !onOpenShare,
      })}
    >
      {onOpenDelete && (!isDownloadDisabled || onOpenShare) && (
        <Button onPress={onImageDelete} size="xl" className={styles.iconButton}>
          <TrashOutline size={24} />
        </Button>
      )}
      {onOpenDelete && isDownloadDisabled && !onOpenShare && (
        <Button onPress={onImageDelete} size="xl" before={<TrashOutline size={24} />}>
          Удалить
        </Button>
      )}
      {!isDownloadDisabled && (
        <Button onPress={onDownload} size="xl" className={styles.iconButton}>
          <DownloadOutline size={24} />
        </Button>
      )}
      {onOpenShare && (
        <Button
          variant="action"
          size="xl"
          before={<ShareArrowOutline size={24} />}
          onPress={onImageShare}
        >
          Поделиться
        </Button>
      )}
    </div>
  );

  return (
    <ImagePresenter
      visible={isActive}
      className={styles.root}
      zIndex={1003}
      keepMounted={false}
      onClose={onClose}
    >
      {searchAppTitle && (
        <ImagePresenter.Title>
          <AppBar onClose={onClose} title={searchAppTitle} state="close" />
        </ImagePresenter.Title>
      )}
      <ImagePresenter.Image>
        <ImageGallery
          ref={galleryRef}
          items={galleryItems}
          showThumbnails={false}
          showPlayButton={false}
          startIndex={startIndex}
          showFullscreenButton={false}
          renderLeftNav={() => <NavLeft galleryRef={galleryRef} />}
          renderRightNav={() => <NavRight galleryRef={galleryRef} />}
        />
      </ImagePresenter.Image>
      <ImagePresenter.BottomControls>{controls}</ImagePresenter.BottomControls>
    </ImagePresenter>
  );
};
