import { ElementType, FC, ReactElement } from 'react';

import { Text } from '@client/shared/ui-kit';
import { PressProps, mergeProps, useHover, usePress } from '@use-platform/react';
import { createSlot, useSlots } from '@yandex-id/components/libs/slots';

import styles from './DocumentThumb.module.css';

export type ColorType = 'orange' | 'green' | 'red' | 'blue' | 'violet';

export interface IDocumentThumb {
  color?: ColorType;
  href?: string;
  icon?: ReactElement;
  as?: ElementType;
}

export interface DescriptionProps {
  error?: boolean;
}

type DocumentThumbProps = IDocumentThumb & PressProps;

const DescriptionSlot = createSlot<DescriptionProps>('DocumentThumb.Description');
const LabelSlot = createSlot('DocumentThumb.Label');
const TitleSlot = createSlot('DocumentThumb.Title');

const _DocumentThumb: FC<DocumentThumbProps> = (props) => {
  const { href, icon, color, children, as, ...otherProps } = props;

  const { isHovered, hoverProps } = useHover(otherProps);
  const { pressProps } = usePress(otherProps);

  const HostElement = as || (href && 'a') || 'div';

  const slots = useSlots(props);
  const label = slots.get(LabelSlot);
  const title = slots.get(TitleSlot);
  const description = slots.get(DescriptionSlot);

  return (
    <HostElement
      tabIndex={0}
      className={styles.root}
      data-color={color}
      data-hovered={isHovered}
      data-new={!description}
      href={href}
      {...mergeProps(hoverProps, pressProps)}
    >
      {icon}
      <div className={styles.info}>
        {label && (
          <Text className={styles.text} as="p" variant="text-xs" color="secondary">
            {label.rendered}
          </Text>
        )}
        {title && (
          <Text weight="medium" as="p" variant="text-s">
            {title.rendered}
          </Text>
        )}
        {description && (
          <Text
            data-error={description.props.error ?? false}
            className={styles.text}
            as="p"
            variant="text-s"
          >
            {description.rendered}
          </Text>
        )}
      </div>
    </HostElement>
  );
};

export const DocumentThumb = Object.assign(_DocumentThumb, {
  Description: DescriptionSlot,
  Label: LabelSlot,
  Title: TitleSlot,
});
