import cx from 'classnames';
import { FC } from 'react';

import { Pencil, Plus, ShareArrowOutline, TrashOutline } from '@client/shared/icons';
import { useEnvContext } from '@client/shared/libs/env';
import { Button, FloatContainer, Modal, Section, SectionStack, Text } from '@client/shared/ui-kit';
import { PressProps } from '@use-platform/react';
import { createSlot, useSlots } from '@yandex-id/components/libs/slots';

import { Color } from './types';
import { FieldItem, FieldItemSkeleton } from './ui/FieldItem';
import { FieldsSection } from './ui/FieldsSection';
import { SafetyHint } from './ui/SafetyHint';

import styles from './DocumentViewer.module.css';

export interface DocumentViewerProps {
  color: Color;
  isVisible: boolean;
  onClose: () => void;
}

export interface BlankFieldData {
  text: string;
  onPress: () => void;
}

export interface BlanksProps {
  elements: BlankFieldData[];
}

export interface IconProps {
  color: Color;
}

export type DeleteProps = PressProps;

export type EditProps = PressProps;

export type ShareProps = PressProps;

const IconSlot = createSlot<IconProps>('DocumentViewer.Icon');
const TitleSlot = createSlot('DocumentViewer.Title');
const SubtitleSlot = createSlot('DocumentViewer.Subtitle');
const PreviewSlot = createSlot('DocumentViewer.Preview');
const BlanksSlot = createSlot<BlanksProps>('DocumentViewer.Blanks');
const DeleteActionSlot = createSlot<DeleteProps>('DocumentViewer.DeleteAction');
const EditActionSlot = createSlot<EditProps>('DocumentViewer.EditAction');
const ShareActionSlot = createSlot<ShareProps>('DocumentViewer.ShareAction');

const _DocumentViewer: FC<DocumentViewerProps> = (props) => {
  const { color, isVisible, onClose, ...restProps } = props;
  const slots = useSlots(restProps);
  const {
    ua: { isYandexApp, isSearchApp },
  } = useEnvContext();

  const icon = slots.get(IconSlot);
  const title = slots.get(TitleSlot);
  const subtitle = slots.get(SubtitleSlot);
  const preview = slots.get(PreviewSlot);
  const blanks = slots.get(BlanksSlot);
  const deleteAction = slots.get(DeleteActionSlot);
  const editAction = slots.get(EditActionSlot);
  const shareAction = slots.get(ShareActionSlot);

  return (
    <>
      <Modal
        data-color={color}
        className={styles.root}
        visible={isVisible}
        keepMounted={false}
        onClose={onClose}
        closeButtonPosition={isSearchApp && isYandexApp ? 'start' : 'end'}
      >
        <div className={styles.container}>
          {icon && (
            <div className={styles.icon} data-color={icon.props.color}>
              {icon.rendered}
            </div>
          )}
          <div className={styles.header}>
            {title && title.rendered}
            {subtitle && subtitle.rendered}
          </div>
          <div className={styles.content}>
            {preview && <div className={styles.preview}>{preview.rendered}</div>}
            <SectionStack className={styles.sectionStack}>
              {slots.children}
              {blanks?.props.elements.length && (
                <Section className={styles.section}>
                  <Text variant="text-m">Осталось заполнить</Text>
                  <div className={styles.blankFields}>
                    {blanks.props.elements.map((element) => (
                      <Button
                        className={styles.blank}
                        key={element.text}
                        onPress={element.onPress}
                        size="m"
                        after={<Plus size={16} />}
                      >
                        <Text
                          className={styles.text}
                          as="span"
                          weight="medium"
                          variant="text-s-long"
                          overflow="ellipsis"
                        >
                          {element.text}
                        </Text>
                      </Button>
                    ))}
                  </div>
                </Section>
              )}
              <Section className={cx(styles.section, styles.section_actions)}>
                <div className={styles.actions}>
                  {deleteAction && (
                    <div className={styles.action}>
                      <Button className={styles.button} onPress={deleteAction.props.onPress}>
                        <TrashOutline />
                      </Button>
                      <Text variant="text-s">Удалить</Text>
                    </div>
                  )}
                  {editAction && (
                    <div className={styles.action}>
                      <Button className={styles.button} onPress={editAction.props.onPress}>
                        <Pencil />
                      </Button>
                      <Text variant="text-s">Изменить</Text>
                    </div>
                  )}
                </div>
                {isYandexApp && isSearchApp && <SafetyHint />}
              </Section>
            </SectionStack>
          </div>
        </div>
      </Modal>
      {shareAction && (
        <FloatContainer>
          <Button variant="action" size="xl" shape="circle" onPress={shareAction.props.onPress}>
            <ShareArrowOutline />
          </Button>
        </FloatContainer>
      )}
    </>
  );
};

export const DocumentViewer = Object.assign(_DocumentViewer, {
  Icon: IconSlot,
  Title: TitleSlot,
  Subtitle: SubtitleSlot,
  Preview: PreviewSlot,
  Blanks: BlanksSlot,
  DeleteAction: DeleteActionSlot,
  EditAction: EditActionSlot,
  ShareAction: ShareActionSlot,
  Section: FieldsSection,
  FieldItem: FieldItem,
  FieldItemSkeleton: FieldItemSkeleton,
});
