/* eslint-disable @yandex-id/i18n/no-unwrapped-strings */
import { useEvent, useStore } from 'effector-react';
import { Key, VFC, useCallback, useEffect } from 'react';

import * as model from '../../models/suggest';
import { GeoSuggestBase } from '../GeoSuggestBase/GeoSuggestBase';
import { GeoSuggestOption } from '../GeoSuggestOption/GeoSuggestOption';

export interface GeoLocation {
  street: string;
  region: string;
  fullText: string;
}

export interface GeoSuggestProps {
  autoFocus?: boolean;
  defaultValue?: string;
  error?: string;
  onBlur?: () => void;
  onChange?: (payload: GeoLocation) => void;
}

export const GeoSuggest: VFC<GeoSuggestProps> = (props) => {
  const { autoFocus, defaultValue, error, onBlur, onChange } = props;

  const onInputChange = useEvent(model.valueChanged);

  const value = useStore(model.$value);
  const addresses = useStore(model.$addresses);

  const onSelectionChange = useCallback(
    (key: Key) => {
      const address = addresses.find((address) => `${address.lat}-${address.lon}` === key);

      if (address) {
        onChange?.({
          region: address.desc,
          street: address.name,
          fullText: `${address.desc}, ${address.name}`,
        });
      }
    },
    [addresses, onChange],
  );

  useEffect(() => {
    if (defaultValue) {
      model.valueChanged(defaultValue);
    }

    return () => {
      model.reset();
    };
  }, [defaultValue]);

  return (
    <GeoSuggestBase
      aria-label="Адрес"
      autoFocus={autoFocus}
      defaultInputValue={defaultValue}
      error={error}
      inputValue={value}
      onBlur={onBlur}
      onInputChange={onInputChange}
      onSelectionChange={onSelectionChange}
    >
      {addresses.map((item) => (
        <GeoSuggestOption
          key={`${item.lat}-${item.lon}`}
          description={item.desc}
          text={item.name}
          textValue={item.name}
        />
      ))}
    </GeoSuggestBase>
  );
};
