import { FC, useEffect, useRef } from 'react';

import { useComboBox } from '@react-aria/combobox';
import { useComboBoxState } from '@react-stately/combobox';
import type { ComboBoxProps } from '@react-types/combobox';

import { ListBox } from '../ListBox/ListBox';
import { Popover } from '../Popover/Popover';
import { SearchField } from '../SearchField/SearchField';

interface GeoSuggestBaseProps extends ComboBoxProps<{}> {
  error?: string;
}

export const GeoSuggestBase: FC<GeoSuggestBaseProps> = (props) => {
  const { error } = props;

  const targetRef = useRef<HTMLDivElement>(null);
  const inputRef = useRef<HTMLInputElement>(null);
  const listBoxRef = useRef<HTMLDivElement>(null);
  const popoverRef = useRef<HTMLDivElement>(null);

  const state = useComboBoxState({ ...props, menuTrigger: 'manual' });

  const { inputProps, listBoxProps } = useComboBox(
    { ...props, inputRef, listBoxRef, popoverRef, shouldFocusWrap: true },
    state,
  );

  useEffect(() => {
    state.setOpen(state.isFocused && state.collection.size > 0);
  }, [state]);

  // NOTE: Используем данное решение для предотвращения полного закрытия диалогового окна,
  // т.к. используем хук overlay из react-aria а не из use-platform.
  useEffect(() => {
    if (!state.isOpen) {
      return;
    }

    const onKeyUp = (event: KeyboardEvent) => {
      if (event.key === 'Escape') {
        event.stopPropagation();
      }
    };

    document.addEventListener('keyup', onKeyUp, true);

    return () => {
      document.removeEventListener('keyup', onKeyUp, true);
    };
  }, [state.isOpen]);

  return (
    <>
      <SearchField
        {...inputProps}
        error={error}
        inputRef={inputRef}
        isFocused={state.isFocused}
        ref={targetRef}
        value={inputProps.value as string}
      />

      <Popover ref={popoverRef} targetRef={targetRef} isVisible={state.isOpen}>
        <ListBox {...listBoxProps} ref={listBoxRef} state={state} />
      </Popover>
    </>
  );
};
