import { VFC } from 'react';

import { logProps } from '@client/shared/libs/metrika';
import { formatCurrency } from '@shared/helpers';

import { Shortcut } from './Shortcut';

import styles from './FamilyPayShortcut.module.css';

interface FamilyPayShortcutProps {
  isAdmin?: boolean;
  hasPay?: boolean;
  cardNumber?: string;
  balance: number;
  currency?: string;
  unlim?: boolean;
}

type FamilyPayShortcutType = 'adminPromo' | 'memberPromo' | 'adminPay' | 'memberPay';

const getFamilyPayShortcutType = (props: FamilyPayShortcutProps): FamilyPayShortcutType => {
  const { isAdmin, hasPay } = props;

  if (isAdmin && !hasPay) return 'adminPromo';
  if (!isAdmin && !hasPay) return 'memberPromo';
  if (isAdmin && hasPay) return 'adminPay';

  return 'memberPay';
};

const types = {
  adminPromo: {
    text: 'Семейная оплата: поделитесь картой с близкими',
    buttonText: 'Попробовать',
    url: 'https://id.yandex.ru/yafamily?utm_source=yandexapp&utm_medium=shortcut&utm_campaign=yafamily&utm_content=for_members',
  },
  adminPay: {
    text: '',
    buttonText: 'Лимиты',
    url: 'https://passport.yandex.ru/profile/family?utm_source=yandexapp&utm_medium=shortcut&utm_campaign=yafamily&utm_content=for_owner',
  },
  memberPromo: {
    text: 'Семейная оплата',
    buttonText: '',
    url: 'https://id.yandex.ru/yafamily?utm_source=yandexapp&utm_medium=shortcut&utm_campaign=yafamily&utm_content=for_members',
  },
  memberPay: {
    text: 'Отстаток на семейной карте',
    buttonText: '',
    url: 'https://passport.yandex.ru/profile/family?utm_source=yandexapp&utm_medium=shortcut&utm_campaign=yafamily&utm_content=for_owner',
  },
};

const getFamilyPayShortcutText = (type: FamilyPayShortcutType, card?: string) => {
  if (type === 'adminPay') {
    return `Семейная карта ••${card}`;
  }

  return types[type].text;
};

const getFamilyPayShortcutButtonText = (
  type: FamilyPayShortcutType,
  balance: number,
  currency?: string,
  unlim?: boolean,
) => {
  if (type === 'memberPay') {
    if (unlim) return 'Безлимит';

    return `${balance} ${formatCurrency(currency)}`.trimEnd();
  }

  return types[type].buttonText;
};

export const FamilyPayShortcut: VFC<FamilyPayShortcutProps> = (props) => {
  const { balance, cardNumber, currency, unlim } = props;
  const type = getFamilyPayShortcutType(props);
  const size = type === 'adminPromo' ? 'l' : 's';

  return (
    <Shortcut
      size={size}
      className={styles.root}
      text={getFamilyPayShortcutText(type, cardNumber)}
      textBold
      withImage
      opacity={type === 'memberPay' ? 'light' : undefined}
      buttonText={getFamilyPayShortcutButtonText(type, balance, currency, unlim)}
      href={types[type].url}
      target="_blank"
      {...logProps('shortcut.familyPay', { variant: type })}
    />
  );
};
