import { useEvent, useGate, useStore } from 'effector-react';
import { VFC } from 'react';

import { useForm } from '@client/shared/libs/effector-forms';
import { capitalizeFirstLetter } from '@client/shared/libs/string';
import {
  Button,
  Form,
  Modal,
  Section,
  SectionStack,
  Select,
  Text,
  TextField,
} from '@client/shared/ui-kit';

import { useAvailableQuestionsQuery } from './api';
import { CUSTOM_QUESTION_ID, EMPTY_QUESTION_ID } from './constants';
import { i18n } from './i18n';
import * as model from './model';

import styles from './ControlQuestionModal.module.css';

export const ControlQuestionUpdateModal: VFC = () => {
  const { fields, submit } = useForm(model.form);

  const isVisible = useStore(model.dialog.$isVisible);
  const isPending = useStore(model.$isPending);

  useGate(model.FormTypeGate, { type: model.FormType.UPDATE, isVisible });

  const { data, loading: isLoading } = useAvailableQuestionsQuery({ skip: !isVisible });

  const handleSubmit = useEvent(submit);
  const handleOnClose = useEvent(model.dialog.hide);

  if (isLoading) {
    // TODO: Показывать загрузку внутри модального окна.
    return null;
  }

  const questions = data?.accountAvailableQuestions;
  const oldQuestion = data?.account.question?.text;

  const selectedQuestionId = fields.selectedQuestionId.value;
  const isEmptyQuestionSelected = selectedQuestionId === EMPTY_QUESTION_ID;
  const isCustomQuestionSelected = selectedQuestionId === CUSTOM_QUESTION_ID;

  return (
    <Modal visible={isVisible} onClose={handleOnClose} keepMounted={false} className={styles.modal}>
      <Form onSubmit={handleSubmit} className={styles.form}>
        <SectionStack className={styles.stack}>
          <Section className={styles.section}>
            <Section.Title isHeading>{i18n('Изменить контрольный вопрос')}</Section.Title>
            <Section.Description>
              {i18n('Введите ответ на старый вопрос:')}
              <br />
              <Text weight="bold" color="inherit">
                «{oldQuestion}»
              </Text>
            </Section.Description>

            <TextField
              autoFocus={isVisible}
              error={fields.oldAnswer.errorText({
                required: i18n('Введите ответ'),
                invalid: i18n('Неверный ответ'),
              })}
              onChangeValue={fields.oldAnswer.onChange}
              placeholder={i18n('Ответ на старый вопрос')}
              value={fields.oldAnswer.value}
            />
          </Section>

          <Section className={styles.section}>
            <Section.Title>{i18n('Новый контрольный вопрос')}</Section.Title>
            <Section.Description>
              {i18n('Выберите вопрос (или придумайте свой) и напишите ответ на него')}
            </Section.Description>

            <Select
              error={fields.selectedQuestionId.errorText({ required: i18n('Выберите вопрос') })}
              onChange={fields.selectedQuestionId.onChange}
              value={fields.selectedQuestionId.value}
            >
              {questions?.map((question) => (
                <Select.Option key={question.id} value={question.id}>
                  {capitalizeFirstLetter(question.text)}
                </Select.Option>
              ))}
            </Select>

            {isCustomQuestionSelected && (
              <TextField
                autoFocus
                error={fields.newQuestion.errorText({
                  required: i18n('Введите вопрос'),
                  maxLength: i18n('Слишком длинный вопрос'),
                })}
                onChangeValue={fields.newQuestion.onChange}
                placeholder={i18n('Введите вопрос')}
                value={fields.newQuestion.value}
              />
            )}

            <TextField
              disabled={isEmptyQuestionSelected}
              error={fields.newAnswer.errorText({
                required: i18n('Введите ответ'),
                maxLength: i18n('Слишком длинный ответ'),
              })}
              label={i18n('Введите ответ')}
              onChangeValue={fields.newAnswer.onChange}
              placeholder={i18n('Ответ на новый вопрос')}
              value={fields.newAnswer.value}
            />

            <Button
              type="submit"
              size="l"
              variant="action"
              onPress={handleSubmit}
              disabled={isPending}
              className={styles.action}
            >
              {isPending ? i18n('Выполняется...') : i18n('Сохранить')}
            </Button>
          </Section>
        </SectionStack>
      </Form>
    </Modal>
  );
};
