import { attach, sample, split } from 'effector';
import { createGate } from 'effector-react';

import { ConfirmByPasswordModel } from '@client/features/confirm-by-password';
import { createForm, rules } from '@client/shared/libs/effector-forms';
import { createDialogApi } from '@client/shared/libs/effector-helpers';

import * as api from './api';
import { CUSTOM_QUESTION_ID, EMPTY_QUESTION_ID } from './constants';
import { notifications } from './libs/notifier';

export enum FormType {
  CREATE,
  UPDATE,
}

interface GateProps {
  type: FormType;
  isVisible: boolean;
}

export const dialog = createDialogApi('control-question');
export const FormTypeGate = createGate<GateProps>();

export const form = createForm({
  fields: {
    type: {
      init: FormType.CREATE,
      units: { $value: FormTypeGate.state.map((state) => state.type) },
    },
    oldAnswer: {
      init: '',
      rules: [rules.requiredWhen((form) => form.type === FormType.UPDATE)],
    },
    selectedQuestionId: {
      init: EMPTY_QUESTION_ID,
      rules: [rules.requiredWhenNotValue(EMPTY_QUESTION_ID)],
    },
    newQuestion: {
      init: '',
      rules: [
        rules.requiredWhen((form) => form.selectedQuestionId === CUSTOM_QUESTION_ID),
        rules.maxLength(37),
      ],
    },
    newAnswer: {
      init: '',
      rules: [
        rules.requiredWhen((form) => form.selectedQuestionId !== EMPTY_QUESTION_ID),
        rules.maxLength(30),
      ],
    },
  },
  units: {
    reset: dialog.hide,
  },
  validateOn: ['blur', 'change', 'submit'],
});

export const setQuestionFx = attach({
  effect: api.setQuestionFx,
  source: {
    newAnswer: form.fields.newAnswer.$value,
    newQuestion: form.fields.newQuestion.$value,
    oldAnswer: form.fields.oldAnswer.$value,
    questionId: form.fields.selectedQuestionId.$value,
    trackId: ConfirmByPasswordModel.form.fields.trackId.$value,
  },
  mapParams: (_, payload) => payload,
});

export const $isPending = setQuestionFx.pending;

split({
  source: setQuestionFx.failData,
  match: (payload) => payload.reason,
  cases: {
    [api.SetQuestionProblemKind.PasswordRequired]:
      ConfirmByPasswordModel.dialog.show.prepend<api.SetQuestionProblem>((payload) => ({
        trackId: payload.trackId,
      })),
    [api.SetQuestionProblemKind.CompareNotMatched]:
      form.fields.oldAnswer.addError.prepend<api.SetQuestionProblem>(() => ({
        rule: 'invalid',
      })),
    [api.SetQuestionProblemKind.Internal]: notifications.internalError,
  },
});

sample({
  clock: form.fields.selectedQuestionId.onChange,
  filter: form.fields.selectedQuestionId.$value.map((value) => value === EMPTY_QUESTION_ID),
  target: form.fields.newAnswer.reset,
});

sample({
  clock: form.fields.selectedQuestionId.onChange,
  filter: form.fields.selectedQuestionId.$value.map((value) => value !== CUSTOM_QUESTION_ID),
  target: form.fields.newQuestion.reset,
});

sample({
  clock: form.formValidated,
  target: setQuestionFx,
});

sample({
  clock: setQuestionFx.done,
  target: dialog.hide,
});

sample({
  clock: ConfirmByPasswordModel.confirmFx.done,
  filter: dialog.$isVisible,
  target: setQuestionFx,
});

sample({
  clock: setQuestionFx.done,
  target: notifications.updatedSuccessful,
});
