import { attach, sample } from 'effector';

import { ConfirmByPasswordModel } from '@client/features/confirm-by-password';
import { createForm, rules } from '@client/shared/libs/effector-forms';
import { createDialogApi } from '@client/shared/libs/effector-helpers/dialog';

import * as api from '../api';
import { createValidationUrl } from '../libs/create-validation-url';
import { notifications } from '../libs/notifier';

export const dialog = createDialogApi('recovery-email');

export const form = createForm({
  fields: {
    email: {
      init: '',
      rules: [rules.required(), rules.email()],
    },
  },
  units: {
    reset: dialog.hide,
  },
});

export const addRecoveryEmailFx = attach({
  effect: api.addRecoveryEmailFx,
  source: {
    email: form.fields.email.$value,
    trackId: ConfirmByPasswordModel.form.fields.trackId.$value,
  },
  mapParams: (_, payload) => ({ ...payload, ...createValidationUrl() }),
});

addRecoveryEmailFx.failData.watch((payload) => {
  switch (payload.reason) {
    case api.AddRecoveryEmailProblemKind.PasswordRequired:
      ConfirmByPasswordModel.dialog.show({ trackId: payload.trackId });
      break;
    case api.AddRecoveryEmailProblemKind.IsNative:
      form.fields.email.addError({ rule: 'native' });
      break;
    case api.AddRecoveryEmailProblemKind.EmailInvalid:
      form.fields.email.addError({ rule: 'email' });
      break;
    case api.AddRecoveryEmailProblemKind.AlreadyConfirmed:
      form.fields.email.addError({ rule: 'confirmed' });
      break;
    case api.AddRecoveryEmailProblemKind.TooManyRequests:
      notifications.tooManyRequests();
      break;
    case api.AddRecoveryEmailProblemKind.Internal:
      notifications.internalError();
      break;
  }
});

export const $isAddEmailPending = addRecoveryEmailFx.pending;

sample({
  clock: form.formValidated,
  target: addRecoveryEmailFx,
});

sample({
  clock: ConfirmByPasswordModel.confirmFx.done,
  filter: dialog.$isVisible,
  target: addRecoveryEmailFx,
});

sample({
  clock: addRecoveryEmailFx.done,
  target: notifications.addedSuccessful,
});
