import { attach, createEvent, merge, sample } from 'effector';

import { ConfirmByPasswordModel } from '@client/features/confirm-by-password';
import { createForm, filters, rules } from '@client/shared/libs/effector-forms';
import { createDialogApi } from '@client/shared/libs/effector-helpers/dialog';

import * as api from '../api';
import * as cookie from '../libs/cookie';
import { createValidationUrl } from '../libs/create-validation-url';
import { notifications } from '../libs/notifier';
import type { Email } from '../types';

interface DialogState {
  email: Email;
  trackId?: string;
}

export const dialog = createDialogApi<DialogState>();

dialog.show.watch((payload) => {
  if (payload.trackId) {
    cookie.setTrackCookie(payload.email, payload.trackId);
  } else {
    const trackId = cookie.getTrackCookie(payload.email);

    if (trackId) {
      dialog.set({ ...payload, trackId });
    }
  }
});

export const codeResendClicked = createEvent();

export const form = createForm({
  fields: {
    code: {
      init: '',
      rules: [rules.requiredLength(6)],
      filter: filters.isNumber,
    },
    trackId: {
      init: '',
      units: {
        $value: dialog.$state.map((state) => state.trackId ?? ''),
      },
    },
  },
  units: {
    reset: merge([dialog.hide, dialog.reset]),
  },
});

export const confirmFx = attach({
  effect: api.confirmRecoveryEmailByCodeFx,
  source: form.$values,
});

export const $isConfirmPending = confirmFx.pending;

confirmFx.failData.watch((payload) => {
  switch (payload.reason) {
    case api.ConfirmRecoveryEmailByCodeProblemKind.IncorrectKey:
      form.fields.code.addError({ rule: 'invalid' });
      break;
    case api.ConfirmRecoveryEmailByCodeProblemKind.AlreadyConfirmed:
      dialog.hide();
      dialog.reset();
      notifications.alreadyConfirmed();
      break;
    case api.ConfirmRecoveryEmailByCodeProblemKind.KeyCheckLimitExceeded:
      notifications.tooManyRequests();
      break;
    case api.ConfirmRecoveryEmailByCodeProblemKind.Internal:
      notifications.internalError();
      break;
  }
});

export const resendFx = attach({
  effect: api.addRecoveryEmailFx,
  source: {
    email: dialog.$state.map((state) => state.email),
  },
  mapParams: (_, payload) => ({ ...payload, ...createValidationUrl() }),
});

export const $isResendPending = resendFx.pending;

resendFx.doneData.watch((trackId) => {
  cookie.setTrackCookie(dialog.$state.getState().email, trackId);
});

resendFx.failData.watch((payload) => {
  switch (payload.reason) {
    case api.AddRecoveryEmailProblemKind.PasswordRequired:
      ConfirmByPasswordModel.dialog.show({ trackId: payload.trackId });
      form.fields.trackId.set(payload.trackId);
      break;
    case api.AddRecoveryEmailProblemKind.TooManyRequests:
      notifications.tooManyRequests();
      break;
    case api.AddRecoveryEmailProblemKind.Internal:
      break;
  }
});

sample({
  clock: form.formValidated,
  target: confirmFx,
});

sample({
  clock: confirmFx.done,
  target: [dialog.hide, dialog.reset, notifications.confirmedSuccessful],
});

sample({
  clock: codeResendClicked,
  target: resendFx,
});

sample({
  clock: resendFx.doneData,
  target: form.fields.trackId.set,
});

sample({
  clock: ConfirmByPasswordModel.confirmFx.done,
  filter: dialog.$isVisible,
  target: resendFx,
});
