/* eslint-disable @yandex-id/i18n/no-unwrapped-strings */
import { useEvent, useStore } from 'effector-react';
import { useCallback, useState } from 'react';

import { CloseConfirmationDialog } from '@client/entities/close-confirmation-dialog';
import { DeleteConfirmationDialog } from '@client/entities/delete-confirmation-dialog';
import { GeoLocation, GeoSuggest } from '@client/entities/geo-suggest';
import ghostSource from '@client/shared/assets/ghost-map.png';
import { TrashOutline } from '@client/shared/icons';
import { useForm } from '@client/shared/libs/effector-forms';
import { Button, Form, Modal, Section, SectionStack, TextField } from '@client/shared/ui-kit';

import { AddressType } from '../../api';
import { closeConfirmationModel, deleteConfirmationModel } from '../../models/confirmation';
import * as updateModel from '../../models/create-or-update-address';
import * as deleteModel from '../../models/delete-address';
import * as formModel from '../../models/form';
import * as rootModel from '../../models/root';
import { AddressPlate } from '../AddressPlate/AddressPlate';

import styles from './AddressEditorDialog.module.css';

export const AddressEditorDialog = () => {
  const [isVisibleSuggest, setVisibleSuggest] = useState(false);
  const { fields, submit, isTouched } = useForm(formModel.form);

  const isEditorMode = Boolean(fields.id.value);
  const shouldForceSuggest = !isEditorMode && !fields.fullText.value;

  const onClose = useEvent(rootModel.closePressed);
  const onSubmit = useEvent(submit);
  const onDelete = useEvent(deleteModel.deletePressed);

  const isVisible = useStore(rootModel.dialog.$isVisible);
  const isCreatePending = useStore(updateModel.$isCreatePending);
  const isDeletePending = useStore(deleteModel.$isDeletePending);

  const onSuggestChangeValue = useCallback(
    (payload: GeoLocation) => {
      setVisibleSuggest(false);
      fields.region.onChange(payload.region);
      fields.street.onChange(payload.street);
      fields.fullText.onChange(payload.fullText);
    },
    [fields.fullText, fields.region, fields.street],
  );

  const onSuggestBlur = useCallback(() => {
    setVisibleSuggest(false);
  }, []);

  const onAddressPlatePress = useCallback(() => {
    setVisibleSuggest(true);
  }, []);

  return (
    <>
      <Modal visible={isVisible} onClose={onClose} keepMounted={false} className={styles.modal}>
        <Form className={styles.content} onSubmit={onSubmit}>
          <SectionStack className={styles.stack}>
            <Section className={styles.section}>
              <Section.Title isHeading>{getDialogTitle(isEditorMode)}</Section.Title>

              {isVisibleSuggest || shouldForceSuggest ? (
                <GeoSuggest
                  autoFocus={isVisibleSuggest || shouldForceSuggest}
                  defaultValue={fields.fullText.value}
                  error={fields.fullText.errorText({ required: 'Укажите адрес' })}
                  onBlur={onSuggestBlur}
                  onChange={onSuggestChangeValue}
                />
              ) : (
                <AddressPlate
                  autoFocus={!isVisibleSuggest}
                  onPress={onAddressPlatePress}
                  region={fields.region.value}
                  text={fields.street.value}
                  type={fields.type.value}
                />
              )}
            </Section>

            <Section className={styles.section}>
              <Section.Title>Поможет при доставке:</Section.Title>

              <div className={styles.fields}>
                <TextField
                  label="Квартира"
                  onChangeValue={fields.room.onChange}
                  size="xl"
                  value={fields.room.value}
                />
                <TextField
                  label="Подъезд"
                  onChangeValue={fields.entrance.onChange}
                  size="xl"
                  value={fields.entrance.value}
                />
                <TextField
                  label="Этаж"
                  onChangeValue={fields.floor.onChange}
                  size="xl"
                  value={fields.floor.value}
                />
                <TextField
                  label="Домофон"
                  onChangeValue={fields.intercom.onChange}
                  size="xl"
                  value={fields.intercom.value}
                />
                <TextField
                  label="Комментарий для курьера"
                  onChangeValue={fields.comment.onChange}
                  size="xl"
                  value={fields.comment.value}
                />
              </div>

              <div className={styles.actions}>
                <Button
                  disabled={!isTouched || isCreatePending}
                  onPress={onSubmit}
                  size="xl"
                  type="submit"
                  variant="action"
                >
                  {isCreatePending ? 'Выполняется...' : 'Сохранить'}
                </Button>
                {isEditorMode && (
                  <Button before={<TrashOutline />} onPress={onDelete} size="xl" variant="clear">
                    {isDeletePending ? 'Выполняется...' : 'Удалить адрес'}
                  </Button>
                )}
              </div>
            </Section>
          </SectionStack>
        </Form>
      </Modal>

      <CloseConfirmationDialog model={closeConfirmationModel} />

      <DeleteConfirmationDialog
        cover={ghostSource}
        model={deleteConfirmationModel}
        text={getDeleteConfirmationText(fields.type.value)}
      />
    </>
  );
};

function getDialogTitle(isEditorMode: boolean) {
  if (isEditorMode) {
    return 'Изменить адрес';
  }

  return 'Добавить адрес';
}

function getDeleteConfirmationText(type: AddressType) {
  return {
    [AddressType.Home]: 'Точно удалить домашний адрес?',
    [AddressType.Work]: 'Точно удалить рабочий адрес?',
    [AddressType.Other]: 'Точно удалить адрес?',
  }[type];
}
