import { ExternalAccountProviderName } from '@client/shared/api/graphql';
import { getSocialBrokerPaths } from '@client/shared/paths';

import {
  CLOSE_EVENT_TYPE,
  LOCAL_STORAGE_KEY,
  REMOVE_ITEM_TIMEOUT,
  RETURN_PATHNAME,
  WINDOW_NAME,
} from './constants';
import { getQueryParams, getWindowProperties, getWindowUrl } from './utils';

interface BindExternalAccountBrokerEvent {
  type: string;
  payload: Record<string, string>;
}

const PROVIDER_NAME_TO_CODE: Record<ExternalAccountProviderName, string> = {
  [ExternalAccountProviderName.Apple]: 'apl',
  [ExternalAccountProviderName.Esia]: 'esia',
  [ExternalAccountProviderName.Facebook]: 'fb',
  [ExternalAccountProviderName.Google]: 'gg',
  [ExternalAccountProviderName.Mailru]: 'mr',
  [ExternalAccountProviderName.Odnoklassniki]: 'ok',
  [ExternalAccountProviderName.Twitter]: 'tw',
  [ExternalAccountProviderName.Vk]: 'vk',
};

class BindExternalAccountBroker {
  private static win: Window | null = null;

  constructor() {
    this.bindByProviderName = this.bindByProviderName.bind(this);
    this.subscribe = this.subscribe.bind(this);
  }

  bindByProviderName(name: ExternalAccountProviderName) {
    const { bindExternalAccountStart } = getSocialBrokerPaths();

    if (BindExternalAccountBroker.win) {
      BindExternalAccountBroker.win.close();
      BindExternalAccountBroker.win = null;
    }

    const code = PROVIDER_NAME_TO_CODE[name];
    const properties = getWindowProperties();
    const url = getWindowUrl(code, {
      url: bindExternalAccountStart,
      retpath: `${location.origin}${RETURN_PATHNAME}`,
      popupName: WINDOW_NAME,
    });

    BindExternalAccountBroker.win = window.open(url, WINDOW_NAME, properties);

    if (BindExternalAccountBroker.win) {
      BindExternalAccountBroker.win.focus();
    }
  }

  subscribe(callback: () => void) {
    function messageListener(event: Pick<MessageEvent, 'origin' | 'data'>) {
      const { data = {} } = event;
      const { type, payload = {} } = data;

      if (
        event.origin !== location.origin ||
        type !== CLOSE_EVENT_TYPE ||
        payload.status !== 'ok'
      ) {
        return;
      }

      callback();
    }

    function storageListener(event: StorageEvent) {
      if (event.isTrusted && event.key === LOCAL_STORAGE_KEY) {
        const json = window.localStorage.getItem(LOCAL_STORAGE_KEY);

        try {
          if (json) {
            const data = JSON.parse(json);

            messageListener({ origin: location.origin, data });
          }
        } catch {}

        setTimeout(() => {
          window.localStorage.removeItem(LOCAL_STORAGE_KEY);
        }, REMOVE_ITEM_TIMEOUT);
      }
    }

    window.addEventListener('message', messageListener);
    window.addEventListener('storage', storageListener);

    return () => {
      window.removeEventListener('message', messageListener);
      window.removeEventListener('storage', storageListener);
    };
  }

  emit() {
    const query = getQueryParams();

    const event: BindExternalAccountBrokerEvent = {
      type: CLOSE_EVENT_TYPE,
      payload: query,
    };

    if (typeof window.opener === 'undefined') {
      try {
        window.localStorage.setItem(LOCAL_STORAGE_KEY, JSON.stringify(event));
      } catch (e) {}
    } else {
      try {
        window.opener.postMessage(event, location.origin);
      } catch (e) {}
    }

    window.close();
  }
}

export const broker = new BindExternalAccountBroker();
