import { useEvent, useGate, useStore } from 'effector-react';
import { VFC, useCallback, useState } from 'react';

import { CloseFormDialog, DocumentForm } from '@client/entities/document-form';
import { DeleteImageDialog, DocumentImageList } from '@client/entities/document-image-list';
import { useForm } from '@client/shared/libs/effector-forms';
import { useQueryParams } from '@client/shared/libs/router';
import { DateField, TextField } from '@client/shared/ui-kit';

import { BirthCertificateFormSkeleton } from './BirthCertificateFormSkeleton';
import * as model from './form-model';
import { FieldType } from './interface';

export interface BirthCertificateFormQueryParams {
  field?: FieldType;
  id?: string;
}

export const BirthCertificateForm: VFC = () => {
  const form = useForm(model.form);
  const queryParams = useQueryParams<BirthCertificateFormQueryParams>();

  const images = useStore(model.$images);

  const isVisible = useStore(model.dialog.$isVisible);
  const isLoading = useStore(model.$isLoading);
  const isPending = useStore(model.$isPending);
  const isImagesUploading = useStore(model.$isImagesUploading);
  const isImageDeleting = useStore(model.$isImageDeleting);
  const isImageViewerVisible = useStore(model.imageViewerDialog.$isVisible);
  const isConfirmationDialogVisible = useStore(model.confirmationDialog.$isVisible);
  const isDeleteImageDialogVisible = useStore(model.deleteImageDialog.$isVisible);

  const handleOnSubmit = useEvent(form.submit);
  const handleOnClose = useEvent(model.closeFormClicked);
  const handleOnCloseConfirm = useEvent(model.confirmCloseFormClicked);
  const handleOnCloseConfirmation = useEvent(model.closeConfirmationDialogCliked);

  const handleOnImageClicked = useEvent(model.imageCliked);
  const handleOnImageViewerClose = useEvent(model.imageViewerDialog.hide);
  const handleOnDeleteImage = useEvent(model.deleteImageClicked);
  const handleOnDeleteImageConfirm = useEvent(model.confirmDeleteImageClicked);
  const handleOnDeleteImageAbort = useEvent(model.abortDeleteImageCliked);

  useGate(model.FormIdGate, {
    id: isVisible ? queryParams.id ?? '' : '',
  });

  const handleOnImagesUpload = useCallback(
    (images: FileList | null) => {
      if (images && images.length > 0) {
        form.fields.images.onChange(images);
      }
    },
    [form.fields.images],
  );

  const [activeImageId, setActiveImageId] = useState<string | null>(null);

  const deleteImage = useCallback(() => {
    if (activeImageId) {
      handleOnDeleteImageConfirm({
        imageId: activeImageId,
      });
    }
  }, [activeImageId, handleOnDeleteImageConfirm]);

  if (!isVisible) {
    return null;
  }

  if (document === null || isLoading) {
    return <BirthCertificateFormSkeleton />;
  }

  return (
    <>
      <DocumentForm
        isVisible={isVisible}
        isDisabled={isPending || !(form.isDirty && form.isValid)}
        onClose={handleOnClose}
        onSubmit={handleOnSubmit}
      >
        <DocumentForm.Title text="Св-во о рождении" />
        <DocumentForm.Preview>
          <DocumentImageList
            onImageClick={handleOnImageClicked}
            onImageViewerClose={handleOnImageViewerClose}
            isImageViewerVisible={isImageViewerVisible}
            onDelete={(id) => {
              setActiveImageId(id);
              handleOnDeleteImage();
            }}
            isUploading={isImagesUploading}
            onUpload={handleOnImagesUpload}
            isDownloadDisabled
            images={images}
            target="upload"
          />
        </DocumentForm.Preview>
        <DocumentForm.Section>
          <DocumentForm.MultiField>
            <TextField
              autoFocus={queryParams.field === 'series'}
              label="Серия и номер"
              placeholder="Серия"
              value={form.fields.series.value}
              onChangeValue={form.fields.series.onChange}
            />
            <TextField
              autoFocus={queryParams.field === 'docNumber'}
              placeholder="Номер"
              inputMode="decimal"
              value={form.fields.docNumber.value}
              onChangeValue={form.fields.docNumber.onChange}
            />
          </DocumentForm.MultiField>
          <DocumentForm.MultiField>
            <TextField
              autoFocus={queryParams.field === 'lastName'}
              label="Фамилия, имя и отчество"
              placeholder="Фамилия"
              value={form.fields.lastName.value}
              onChangeValue={form.fields.lastName.onChange}
            />
            <TextField
              autoFocus={queryParams.field === 'firstName'}
              placeholder="Имя"
              value={form.fields.firstName.value}
              onChangeValue={form.fields.firstName.onChange}
            />
            <TextField
              autoFocus={queryParams.field === 'middleName'}
              placeholder="Отчество"
              value={form.fields.middleName.value}
              onChangeValue={form.fields.middleName.onChange}
            />
          </DocumentForm.MultiField>
          <DateField
            size="xl"
            label="Дата рождения"
            autoFocus={queryParams.field === 'birthDate'}
            value={form.fields.birthDate.value}
            onChangeValue={form.fields.birthDate.onChange}
          />
          <TextField
            autoFocus={queryParams.field === 'birthPlace'}
            label="Место рождения"
            value={form.fields.birthPlace.value}
            onChangeValue={form.fields.birthPlace.onChange}
          />
          <TextField
            autoFocus={queryParams.field === 'birthAkt'}
            label="Номер записи акта о рождении"
            inputMode="decimal"
            value={form.fields.birthAkt.value}
            onChangeValue={form.fields.birthAkt.onChange}
          />
        </DocumentForm.Section>
        <DocumentForm.Section>
          <DocumentForm.MultiField>
            <TextField
              autoFocus={queryParams.field === 'fatherLastName'}
              label="ФИО отца"
              placeholder="Фамилия"
              value={form.fields.fatherLastName.value}
              onChangeValue={form.fields.fatherLastName.onChange}
            />
            <TextField
              autoFocus={queryParams.field === 'fatherFirstName'}
              placeholder="Имя"
              value={form.fields.fatherFirstName.value}
              onChangeValue={form.fields.fatherFirstName.onChange}
            />
            <TextField
              autoFocus={queryParams.field === 'fatherMiddleName'}
              placeholder="Отчество"
              value={form.fields.fatherMiddleName.value}
              onChangeValue={form.fields.fatherMiddleName.onChange}
            />
          </DocumentForm.MultiField>
          <TextField
            autoFocus={queryParams.field === 'fatherCitizenship'}
            label="Гражданство отца"
            value={form.fields.fatherCitizenship.value}
            onChangeValue={form.fields.fatherCitizenship.onChange}
          />
          <DateField
            size="xl"
            label="Дата рождения отца"
            autoFocus={queryParams.field === 'fatherBirthDate'}
            value={form.fields.fatherBirthDate.value}
            onChangeValue={form.fields.fatherBirthDate.onChange}
          />
          <TextField
            autoFocus={queryParams.field === 'fatherNationality'}
            label="Национальность отца"
            value={form.fields.fatherNationality.value}
            onChangeValue={form.fields.fatherNationality.onChange}
          />
        </DocumentForm.Section>
        <DocumentForm.Section>
          <DocumentForm.MultiField>
            <TextField
              autoFocus={queryParams.field === 'motherLastName'}
              label="ФИО матери"
              placeholder="Фамилия"
              value={form.fields.motherLastName.value}
              onChangeValue={form.fields.motherLastName.onChange}
            />
            <TextField
              autoFocus={queryParams.field === 'motherFirstName'}
              placeholder="Имя"
              value={form.fields.motherFirstName.value}
              onChangeValue={form.fields.motherFirstName.onChange}
            />
            <TextField
              autoFocus={queryParams.field === 'motherMiddleName'}
              placeholder="Отчество"
              value={form.fields.motherMiddleName.value}
              onChangeValue={form.fields.motherMiddleName.onChange}
            />
          </DocumentForm.MultiField>
          <TextField
            autoFocus={queryParams.field === 'motherCitizenship'}
            label="Гражданство матери"
            value={form.fields.motherCitizenship.value}
            onChangeValue={form.fields.motherCitizenship.onChange}
          />
          <DateField
            size="xl"
            label="Дата рождения матери"
            autoFocus={queryParams.field === 'motherBirthDate'}
            value={form.fields.motherBirthDate.value}
            onChangeValue={form.fields.motherBirthDate.onChange}
          />
          <TextField
            autoFocus={queryParams.field === 'motherNationality'}
            label="Национальность матери"
            value={form.fields.motherNationality.value}
            onChangeValue={form.fields.motherNationality.onChange}
          />
        </DocumentForm.Section>
        <DocumentForm.Section>
          <TextField
            autoFocus={queryParams.field === 'registrationPlace'}
            label="Место государственной регистрации"
            value={form.fields.registrationPlace.value}
            onChangeValue={form.fields.registrationPlace.onChange}
          />
          <TextField
            autoFocus={queryParams.field === 'issuePlace'}
            label="Место выдачи свидетельства"
            value={form.fields.issuePlace.value}
            onChangeValue={form.fields.issuePlace.onChange}
          />
          <DateField
            size="xl"
            label="Дата выдачи"
            autoFocus={queryParams.field === 'issueDate'}
            value={form.fields.issueDate.value}
            onChangeValue={form.fields.issueDate.onChange}
          />
        </DocumentForm.Section>
      </DocumentForm>
      <CloseFormDialog
        isVisible={isConfirmationDialogVisible}
        onCloseDialog={handleOnCloseConfirmation}
        onContinueEditing={handleOnCloseConfirmation}
        onCloseForm={handleOnCloseConfirm}
      />
      <DeleteImageDialog
        isVisible={isDeleteImageDialogVisible}
        disabled={isImageDeleting}
        onConfirm={deleteImage}
        onAbort={handleOnDeleteImageAbort}
      />
    </>
  );
};
