import Axios, { AxiosResponse } from 'axios';
import { createEffect } from 'effector';

import {
  AddBirthCertificateDocument,
  AddBirthCertificateMutation,
  AddBirthCertificateMutationVariables,
  AddDocumentProblemReason,
  BirthCertificateDocument,
  DeleteBirthCertificateDocumentDocument,
  DeleteBirthCertificateDocumentMutation,
  DeleteBirthCertificateDocumentMutationVariables,
  DeleteDocumentInput,
  DeleteDocumentProblemReason,
  DocumentByIdProblemReason,
  DocumentImage,
  DocumentImageInput,
  DocumentType,
  GetBirthCertificateByIdDocument,
  GetBirthCertificateByIdQuery,
  GetBirthCertificateByIdQueryVariables,
  UpdateBirthCertificateDocument,
  UpdateBirthCertificateMutation,
  UpdateBirthCertificateMutationVariables,
  UpdateDocumentProblemReason,
} from '@client/shared/api/graphql';
import { getOrCreateClient } from '@client/shared/libs/apollo';

import { MAX_FILE_SIZE_BYTES } from '../../../constants';
import { formatDateToISO } from '../../../libs/date';
import { BirthCertificateFormFields } from '../interface';
import { Field } from '../types/field';

const { request } = Axios.create({
  baseURL: '/personal/api/',
  // TODO: вот тут вопрос
  // сколько ставить
  // а еще мб как-нибудь ограничить рамер файла?
  timeout: 10000,
});

export interface DocumentId {
  id?: string;
}

export interface SaveBirthCertificateFormDataFxInput {
  formData: BirthCertificateFormFields;
  images: DocumentImage[];
}

export interface UploadImagesFxInput {
  formData: BirthCertificateFormFields;
  images: FileList | null;
}

export interface UploadAndSaveImagesFxInput {
  id: string;
  images: FileList;
}

export interface DeleteImageFxInput {
  document: BirthCertificateDocument;
  imageId: string;
}

interface DeleteDocumentError {
  reason: DeleteDocumentProblemReason;
}

interface DeleteDocumentError {
  reason: DeleteDocumentProblemReason;
}

interface GetBirthCertificateError {
  reason: DocumentByIdProblemReason;
}

interface TemporaryDeleteImageFxInput {
  images: DocumentImage[];
  imageId: string;
}

export const saveBirthCertificateFormDataFx = createEffect(
  async (input: SaveBirthCertificateFormDataFxInput) => {
    const { formData, images } = input;

    if (formData.id) {
      const response = await updateBirthCertificate({
        id: formData.id,
        images,
        series: formData.series,
        docNumber: formData.docNumber,
        firstName: formData.firstName,
        lastName: formData.lastName,
        middleName: formData.middleName,
        birthDate: formatDateToISO(formData.birthDate),
        birthPlace: formData.birthPlace,
        birthAkt: formData.birthAkt,
        fatherFirstName: formData.fatherFirstName,
        fatherLastName: formData.fatherLastName,
        fatherMiddleName: formData.fatherMiddleName,
        fatherCitizenship: formData.fatherCitizenship,
        fatherBirthDate: formatDateToISO(formData.fatherBirthDate),
        fatherNationality: formData.fatherNationality,
        motherFirstName: formData.motherFirstName,
        motherLastName: formData.motherLastName,
        motherMiddleName: formData.motherMiddleName,
        motherCitizenship: formData.motherCitizenship,
        motherBirthDate: formatDateToISO(formData.motherBirthDate),
        motherNationality: formData.motherNationality,
        registrationPlace: formData.registrationPlace,
        issuePlace: formData.issuePlace,
        issueDate: formatDateToISO(formData.issueDate),
      });

      if (response.status === 'error') {
        return Promise.reject({ reason: response.reason });
      }
    } else {
      const response = await addBirthCertificate({
        images,
        series: formData.series,
        docNumber: formData.docNumber,
        firstName: formData.firstName,
        lastName: formData.lastName,
        middleName: formData.middleName,
        birthDate: formatDateToISO(formData.birthDate),
        birthPlace: formData.birthPlace,
        birthAkt: formData.birthAkt,
        fatherFirstName: formData.fatherFirstName,
        fatherLastName: formData.fatherLastName,
        fatherMiddleName: formData.fatherMiddleName,
        fatherCitizenship: formData.fatherCitizenship,
        fatherBirthDate: formatDateToISO(formData.fatherBirthDate),
        fatherNationality: formData.fatherNationality,
        motherFirstName: formData.motherFirstName,
        motherLastName: formData.motherLastName,
        motherMiddleName: formData.motherMiddleName,
        motherCitizenship: formData.motherCitizenship,
        motherBirthDate: formatDateToISO(formData.motherBirthDate),
        motherNationality: formData.motherNationality,
        registrationPlace: formData.registrationPlace,
        issuePlace: formData.issuePlace,
        issueDate: formatDateToISO(formData.issueDate),
      });

      if (response.status === 'error') {
        return Promise.reject({ reason: response.reason });
      }
    }
  },
);

export const deleteBirthCertificateFx = createEffect<
  DeleteDocumentInput,
  string,
  DeleteDocumentError
>(async (input) => {
  const apollo = getOrCreateClient({});

  const result = await apollo.mutate<
    DeleteBirthCertificateDocumentMutation,
    DeleteBirthCertificateDocumentMutationVariables
  >({
    mutation: DeleteBirthCertificateDocumentDocument,
    variables: {
      input,
    },
  });

  if (!result.data || result.data.deleteDocument.__typename === 'DeleteDocumentProblem') {
    return Promise.reject({
      reason: DeleteDocumentProblemReason.Internal,
    });
  }

  return 'ok';
});

export const getBirthCertificateDataFx = createEffect<
  DocumentId,
  BirthCertificateDocument | null,
  GetBirthCertificateError
>(async (input) => {
  if (input.id) {
    const apollo = getOrCreateClient({});

    const result = await apollo.query<
      GetBirthCertificateByIdQuery,
      GetBirthCertificateByIdQueryVariables
    >({
      query: GetBirthCertificateByIdDocument,
      variables: {
        id: input.id,
      },
    });

    if (
      result.data.document.__typename === 'BirthCertificateDocument' &&
      result.data.document.type === DocumentType.BirthCertificate
    ) {
      return result.data.document;
    }

    return Promise.reject({
      reason: DocumentByIdProblemReason.NotFound,
    });
  }

  return null;
});

export const uploadImagesFx = createEffect(async ({ images, formData }: UploadImagesFxInput) => {
  if (images) {
    const responses = await uploadImages<DocumentImage>(images, formData.id ?? '', false);

    const uploadedImages = responses.map((response) => response.data);

    return uploadedImages;
  }

  return [];
});

export const uploadAndSaveImagesFx = createEffect<
  UploadAndSaveImagesFxInput,
  BirthCertificateDocument
>(async (input) => {
  const { images: files, id } = input;

  await uploadImages<Document>(files, id, true);

  const apollo = getOrCreateClient({});

  const result = await apollo.query<
    GetBirthCertificateByIdQuery,
    GetBirthCertificateByIdQueryVariables
  >({
    query: GetBirthCertificateByIdDocument,
    fetchPolicy: 'network-only',
    variables: {
      id,
    },
  });

  if (result.data.document.__typename !== 'BirthCertificateDocument') {
    return Promise.reject({});
  }

  return result.data.document;
});

export const deleteImageFx = createEffect<
  DeleteImageFxInput,
  BirthCertificateDocument,
  { response: any }
>(async (input) => {
  const { document, imageId } = input;

  const images = document.images.filter((image) => image.id !== imageId);

  const response = await updateBirthCertificate({
    ...document,
    images,
  });

  if (
    response.status === 'error' ||
    !response.data ||
    response.data.__typename !== 'BirthCertificateDocument'
  ) {
    return Promise.reject({ reason: response.reason });
  }

  return response.data;
});

export const temporaryDeleteImageFx = createEffect((input: TemporaryDeleteImageFxInput) => {
  const { images, imageId } = input;
  const newImages = images.filter((image) => image.id !== imageId);

  return newImages;
});

export const shareDocumentTextFx = createEffect<Field[], void, void>(async (fields) => {
  // Надо оборачивать в трайкетч, иначе отмена шаринга бросает исключение
  try {
    await navigator.share({
      title: 'Свидетельство о рождении',
      text: fields.reduce((acc, value) => acc + `${value.label}: ${value.text}\n`, ''),
    });
  } catch (e) {}
});

const shareImages = async (images: DocumentImage[]) => {
  const files: File[] = [];

  for (let i = 0; i < images.length; i++) {
    const { originalUrl } = images[i];
    const path = new URL(originalUrl).pathname;
    const response = await request({
      url: 'avatars-proxy',
      params: { path },
      responseType: 'arraybuffer',
    });

    const type = response.headers['content-type'];
    const [_, extension] = type.split('/');

    files.push(new File([new Uint8Array(response.data)], `image${i + 1}.${extension}`, { type }));
  }

  // Надо оборачивать в трайкетч, иначе отмена шаринга бросает исключение
  try {
    await navigator.share({
      files,
    });
  } catch (e) {}
};

export const shareImagesFx = createEffect(shareImages);

export const shareImageFx = createEffect<DocumentImage, void, void>(async (image) => {
  return shareImages([image]);
});

async function addBirthCertificate(
  document: Omit<BirthCertificateDocument, 'id' | 'section' | 'type'>,
) {
  const apollo = getOrCreateClient({});

  const images: DocumentImageInput[] = document.images.map((image) => ({
    id: image.id,
    originalHeight: image.originalHeight,
    originalUri: image.originalUri,
    originalUrl: image.originalUrl,
    originalWidth: image.originalWidth,
    previewHeight: image.previewHeight,
    previewUri: image.previewUri,
    previewUrl: image.previewUrl,
    previewWidth: image.previewWidth,
  }));

  const result = await apollo.mutate<
    AddBirthCertificateMutation,
    AddBirthCertificateMutationVariables
  >({
    mutation: AddBirthCertificateDocument,
    variables: {
      input: {
        type: DocumentType.BirthCertificate,
        birthCertificateInput: {
          images,
          series: document.series,
          docNumber: document.docNumber,
          firstName: document.firstName,
          lastName: document.lastName,
          middleName: document.middleName,
          birthDate: document.birthDate,
          birthPlace: document.birthPlace,
          birthAkt: document.birthAkt,
          fatherFirstName: document.fatherFirstName,
          fatherLastName: document.fatherLastName,
          fatherMiddleName: document.fatherMiddleName,
          fatherCitizenship: document.fatherCitizenship,
          fatherBirthDate: document.fatherBirthDate,
          fatherNationality: document.fatherNationality,
          motherFirstName: document.motherFirstName,
          motherLastName: document.motherLastName,
          motherMiddleName: document.motherMiddleName,
          motherCitizenship: document.motherCitizenship,
          motherBirthDate: document.motherBirthDate,
          motherNationality: document.motherNationality,
          registrationPlace: document.registrationPlace,
          issuePlace: document.issuePlace,
          issueDate: document.issueDate,
        },
      },
    },
  });

  if (!result.data || result.data.addDocument.__typename === 'AddDocumentProblem') {
    return {
      status: 'error',
      reason: AddDocumentProblemReason.Internal,
    };
  }

  return {
    status: 'ok',
  };
}

async function updateBirthCertificate(
  document: Omit<BirthCertificateDocument, 'section' | 'type'>,
) {
  const apollo = getOrCreateClient({});

  const images: DocumentImageInput[] = document.images.map((image) => ({
    id: image.id,
    originalHeight: image.originalHeight,
    originalUri: image.originalUri,
    originalUrl: image.originalUrl,
    originalWidth: image.originalWidth,
    previewHeight: image.previewHeight,
    previewUri: image.previewUri,
    previewUrl: image.previewUrl,
    previewWidth: image.previewWidth,
  }));

  const result = await apollo.mutate<
    UpdateBirthCertificateMutation,
    UpdateBirthCertificateMutationVariables
  >({
    mutation: UpdateBirthCertificateDocument,
    variables: {
      input: {
        type: DocumentType.BirthCertificate,
        birthCertificateInput: {
          id: document.id,
          verificationStatus: document.verificationStatus,
          images,
          series: document.series,
          docNumber: document.docNumber,
          firstName: document.firstName,
          lastName: document.lastName,
          middleName: document.middleName,
          birthDate: document.birthDate,
          birthPlace: document.birthPlace,
          birthAkt: document.birthAkt,
          fatherFirstName: document.fatherFirstName,
          fatherLastName: document.fatherLastName,
          fatherMiddleName: document.fatherMiddleName,
          fatherCitizenship: document.fatherCitizenship,
          fatherBirthDate: document.fatherBirthDate,
          fatherNationality: document.fatherNationality,
          motherFirstName: document.motherFirstName,
          motherLastName: document.motherLastName,
          motherMiddleName: document.motherMiddleName,
          motherCitizenship: document.motherCitizenship,
          motherBirthDate: document.motherBirthDate,
          motherNationality: document.motherNationality,
          registrationPlace: document.registrationPlace,
          issuePlace: document.issuePlace,
          issueDate: document.issueDate,
        },
      },
      id: document.id,
    },
  });

  if (!result.data || result.data.updateDocument.__typename === 'UpdateDocumentProblem') {
    return {
      status: 'error',
      reason: UpdateDocumentProblemReason.Internal,
    };
  }

  return {
    status: 'ok',
    data:
      result.data.updateDocument.__typename === 'UpdateDocumentPayload' &&
      result.data.updateDocument.query.document,
  };
}

async function uploadImage<T>(file: File, id?: string, save = true) {
  const form = new FormData();

  form.append('file', file);
  form.append('type', DocumentType.BirthCertificate);
  form.append('save', save ? 'true' : 'false');

  if (id) {
    form.append('id', id);
  }

  const response = await request<T>({
    url: `/documents/upload`,
    method: 'POST',
    data: form,
  });

  return response;
}

async function uploadImages<T>(files: FileList, id: string, save = true) {
  const uploadRequests = Object.values(files).reduce<Promise<AxiosResponse<T, any>>[]>(
    (acc, file) => {
      if (file.size < MAX_FILE_SIZE_BYTES) {
        const uploadDocumentData = uploadImage<T>(file, id, save);

        acc.push(uploadDocumentData);
      }

      return acc;
    },
    [],
  );

  return await Promise.all(uploadRequests);
}
