import { useEvent, useGate, useStore } from 'effector-react';
import { VFC, useCallback, useState } from 'react';

import { CloseFormDialog, DocumentForm } from '@client/entities/document-form';
import { DeleteImageDialog, DocumentImageList } from '@client/entities/document-image-list';
import { useForm } from '@client/shared/libs/effector-forms';
import { useQueryParams } from '@client/shared/libs/router';
import { DateField, TextField } from '@client/shared/ui-kit';

import { DriverLicenseFormSkeleton } from './DriverLicenseFormSkeleton';
import * as model from './form-model';
import { FieldType } from './interface';

export interface DriverLicenseFormQueryParams {
  field?: FieldType;
  id?: string;
}

export const DriverLicenseForm: VFC = () => {
  const form = useForm(model.form);
  const queryParams = useQueryParams<DriverLicenseFormQueryParams>();

  const images = useStore(model.$images);

  const isVisible = useStore(model.dialog.$isVisible);
  const isLoading = useStore(model.$isLoading);
  const isPending = useStore(model.$isPending);
  const isImagesUploading = useStore(model.$isImagesUploading);
  const isImageDeleting = useStore(model.$isImageDeleting);
  const isImageViewerVisible = useStore(model.imageViewerDialog.$isVisible);
  const isConfirmationDialogVisible = useStore(model.confirmationDialog.$isVisible);
  const isDeleteImageDialogVisible = useStore(model.deleteImageDialog.$isVisible);

  const handleOnSubmit = useEvent(form.submit);
  const handleOnClose = useEvent(model.closeFormClicked);
  const handleOnCloseConfirm = useEvent(model.confirmCloseFormClicked);
  const handleOnCloseConfirmation = useEvent(model.closeConfirmationDialogCliked);

  const handleOnImageClicked = useEvent(model.imageCliked);
  const handleOnImageViewerClose = useEvent(model.imageViewerDialog.hide);
  const handleOnDeleteImage = useEvent(model.deleteImageClicked);
  const handleOnDeleteImageConfirm = useEvent(model.confirmDeleteImageClicked);
  const handleOnDeleteImageAbort = useEvent(model.abortDeleteImageCliked);

  useGate(model.FormIdGate, {
    id: isVisible ? queryParams.id ?? '' : '',
  });

  const handleOnImagesUpload = useCallback(
    (images: FileList | null) => {
      if (images && images.length > 0) {
        form.fields.images.onChange(images);
      }
    },
    [form.fields.images],
  );

  const [activeImageId, setActiveImageId] = useState<string | null>(null);

  const deleteImage = useCallback(() => {
    if (activeImageId) {
      handleOnDeleteImageConfirm({
        imageId: activeImageId,
      });
    }
  }, [activeImageId, handleOnDeleteImageConfirm]);

  if (!isVisible) {
    return null;
  }

  if (document === null || isLoading) {
    return <DriverLicenseFormSkeleton />;
  }

  return (
    <>
      <DocumentForm
        isVisible={isVisible}
        isDisabled={isPending || !(form.isDirty && form.isValid)}
        onClose={handleOnClose}
        onSubmit={handleOnSubmit}
      >
        <DocumentForm.Title text="ВУ" />
        <DocumentForm.Preview>
          <DocumentImageList
            onImageClick={handleOnImageClicked}
            onImageViewerClose={handleOnImageViewerClose}
            isImageViewerVisible={isImageViewerVisible}
            onDelete={(id) => {
              setActiveImageId(id);
              handleOnDeleteImage();
            }}
            isUploading={isImagesUploading}
            onUpload={handleOnImagesUpload}
            isDownloadDisabled
            images={images}
            target="upload"
          />
        </DocumentForm.Preview>
        <DocumentForm.Section>
          <DocumentForm.MultiField>
            <TextField
              autoFocus={queryParams.field === 'lastName'}
              label="Фамилия"
              placeholder="Фамилия"
              value={form.fields.lastName.value}
              onChangeValue={form.fields.lastName.onChange}
            />
            <TextField
              placeholder="Фамилия латиницей"
              value={form.fields.lastNameLat.value}
              onChangeValue={form.fields.lastNameLat.onChange}
            />
          </DocumentForm.MultiField>
          <DocumentForm.MultiField>
            <TextField
              autoFocus={queryParams.field === 'firstName'}
              label="Имя"
              placeholder="Имя"
              value={form.fields.firstName.value}
              onChangeValue={form.fields.firstName.onChange}
            />
            <TextField
              placeholder="Имя латиницей"
              value={form.fields.firstNameLat.value}
              onChangeValue={form.fields.firstNameLat.onChange}
            />
          </DocumentForm.MultiField>
          <DocumentForm.MultiField>
            <TextField
              autoFocus={queryParams.field === 'middleName'}
              label="Отчество"
              placeholder="Отчество"
              value={form.fields.middleName.value}
              onChangeValue={form.fields.middleName.onChange}
            />
            <TextField
              placeholder="Отчество латиницей"
              value={form.fields.middleNameLat.value}
              onChangeValue={form.fields.middleNameLat.onChange}
            />
          </DocumentForm.MultiField>
          <DateField
            size="xl"
            label="Дата рождения"
            autoFocus={queryParams.field === 'birthDate'}
            value={form.fields.birthDate.value}
            onChangeValue={form.fields.birthDate.onChange}
          />
          <DocumentForm.MultiField>
            <TextField
              autoFocus={queryParams.field === 'birthPlace'}
              label="Место рождения"
              placeholder="Место рождения"
              value={form.fields.birthPlace.value}
              onChangeValue={form.fields.birthPlace.onChange}
            />
            <TextField
              placeholder="Место рождения латиницей"
              value={form.fields.birthPlaceLat.value}
              onChangeValue={form.fields.birthPlaceLat.onChange}
            />
          </DocumentForm.MultiField>
        </DocumentForm.Section>
        <DocumentForm.Section>
          <DateField
            size="xl"
            label="Дата выдачи"
            autoFocus={queryParams.field === 'issueDate'}
            value={form.fields.issueDate.value}
            onChangeValue={form.fields.issueDate.onChange}
          />
          <DateField
            size="xl"
            label="Действует до"
            autoFocus={queryParams.field === 'endDate'}
            value={form.fields.endDate.value}
            onChangeValue={form.fields.endDate.onChange}
          />
          <DocumentForm.MultiField>
            <TextField
              autoFocus={queryParams.field === 'issuedBy'}
              label="Кем выдано"
              placeholder="Кем выдано"
              value={form.fields.issuedBy.value}
              onChangeValue={form.fields.issuedBy.onChange}
            />
            <TextField
              placeholder="Кем выдано латиницей"
              value={form.fields.issuedByLat.value}
              onChangeValue={form.fields.issuedByLat.onChange}
            />
          </DocumentForm.MultiField>
          <TextField
            autoFocus={queryParams.field === 'docNumber'}
            label="Номер"
            inputMode="decimal"
            value={form.fields.docNumber.value}
            error={form.fields.docNumber.errorText({
              regexp: 'Введите номер полностью (10 цифр)',
            })}
            onBlur={() => form.fields.docNumber.onBlur()}
            onChangeValue={form.fields.docNumber.onChange}
          />
          <DocumentForm.MultiField>
            <TextField
              autoFocus={queryParams.field === 'issuedCity'}
              label="Место выдачи"
              placeholder="Место выдачи"
              value={form.fields.issuedCity.value}
              onChangeValue={form.fields.issuedCity.onChange}
            />
            <TextField
              placeholder="Место выдачи латиницей"
              value={form.fields.issuedCityLat.value}
              onChangeValue={form.fields.issuedCityLat.onChange}
            />
          </DocumentForm.MultiField>
        </DocumentForm.Section>
        <DocumentForm.Section>
          <TextField
            autoFocus={queryParams.field === 'category'}
            label="Категории"
            value={form.fields.category.value}
            onChangeValue={form.fields.category.onChange}
          />
          <TextField
            autoFocus={queryParams.field === 'additionalInformation'}
            label="Особые отметки"
            value={form.fields.additionalInformation.value}
            onChangeValue={form.fields.additionalInformation.onChange}
          />
        </DocumentForm.Section>
      </DocumentForm>
      <CloseFormDialog
        isVisible={isConfirmationDialogVisible}
        onCloseDialog={handleOnCloseConfirmation}
        onContinueEditing={handleOnCloseConfirmation}
        onCloseForm={handleOnCloseConfirm}
      />
      <DeleteImageDialog
        isVisible={isDeleteImageDialogVisible}
        disabled={isImageDeleting}
        onConfirm={deleteImage}
        onAbort={handleOnDeleteImageAbort}
      />
    </>
  );
};
