import Axios, { AxiosResponse } from 'axios';
import { createEffect } from 'effector';

import {
  AddDocumentProblemReason,
  AddDriverLicenseDocument,
  AddDriverLicenseMutation,
  AddDriverLicenseMutationVariables,
  DeleteDocumentInput,
  DeleteDocumentProblemReason,
  DeleteDriverLicenseDocumentDocument,
  DeleteDriverLicenseDocumentMutation,
  DeleteDriverLicenseDocumentMutationVariables,
  DocumentByIdProblemReason,
  DocumentImage,
  DocumentImageInput,
  DocumentType,
  DriverLicenseDocument,
  GetDriverLicenseByIdDocument,
  GetDriverLicenseByIdQuery,
  GetDriverLicenseByIdQueryVariables,
  UpdateDocumentProblemReason,
  UpdateDriverLicenseDocument,
  UpdateDriverLicenseMutation,
  UpdateDriverLicenseMutationVariables,
} from '@client/shared/api/graphql';
import { getOrCreateClient } from '@client/shared/libs/apollo';

import { MAX_FILE_SIZE_BYTES } from '../../../constants';
import { formatDateToISO } from '../../../libs/date';
import { DriverLicenseFormFields } from '../interface';
import { Field } from '../types/field';

const { request } = Axios.create({
  baseURL: '/personal/api/',
  // TODO: вот тут вопрос
  // сколько ставить
  // а еще мб как-нибудь ограничить рамер файла?
  timeout: 10000,
});

export interface DocumentId {
  id?: string;
}

export interface SaveDriverLicenseFormDataFxInput {
  formData: DriverLicenseFormFields;
  images: DocumentImage[];
}

export interface UploadImagesFxInput {
  formData: DriverLicenseFormFields;
  images: FileList | null;
}

export interface UploadAndSaveImagesFxInput {
  id: string;
  images: FileList;
}

export interface DeleteImageFxInput {
  document: DriverLicenseDocument;
  imageId: string;
}

interface DeleteDocumentError {
  reason: DeleteDocumentProblemReason;
}

interface GetDriverLicenseError {
  reason: DocumentByIdProblemReason;
}

interface TemporaryDeleteImageFxInput {
  images: DocumentImage[];
  imageId: string;
}

export const saveDriverLicenseFormDataFx = createEffect(
  async (input: SaveDriverLicenseFormDataFxInput) => {
    const { formData, images } = input;

    if (formData.id) {
      const response = await updateDriverLicense({
        id: formData.id,
        images,
        lastName: formData.lastName,
        lastNameLat: formData.lastNameLat,
        firstName: formData.firstName,
        firstNameLat: formData.firstNameLat,
        middleName: formData.middleName,
        middleNameLat: formData.middleNameLat,
        birthDate: formatDateToISO(formData.birthDate),
        birthPlace: formData.birthPlace,
        birthPlaceLat: formData.birthPlaceLat,
        issueDate: formatDateToISO(formData.issueDate),
        endDate: formatDateToISO(formData.endDate),
        issuedBy: formData.issuedBy,
        issuedByLat: formData.issuedByLat,
        docNumber: formData.docNumber,
        issuedCity: formData.issuedCity,
        issuedCityLat: formData.issuedCityLat,
        category: formData.category,
        additionalInformation: formData.additionalInformation,
      });

      if (response.status === 'error') {
        return Promise.reject({ reason: response.reason });
      }
    } else {
      const response = await addDriverLicense({
        images,
        lastName: formData.lastName,
        lastNameLat: formData.lastNameLat,
        firstName: formData.firstName,
        firstNameLat: formData.firstNameLat,
        middleName: formData.middleName,
        middleNameLat: formData.middleNameLat,
        birthDate: formatDateToISO(formData.birthDate),
        birthPlace: formData.birthPlace,
        birthPlaceLat: formData.birthPlaceLat,
        issueDate: formatDateToISO(formData.issueDate),
        endDate: formatDateToISO(formData.endDate),
        issuedBy: formData.issuedBy,
        issuedByLat: formData.issuedByLat,
        docNumber: formData.docNumber,
        issuedCity: formData.issuedCity,
        issuedCityLat: formData.issuedCityLat,
        category: formData.category,
        additionalInformation: formData.additionalInformation,
      });

      if (response.status === 'error') {
        return Promise.reject({ reason: response.reason });
      }
    }
  },
);

export const deleteDriverLicenseFx = createEffect<DeleteDocumentInput, string, DeleteDocumentError>(
  async (input) => {
    const apollo = getOrCreateClient({});

    const result = await apollo.mutate<
      DeleteDriverLicenseDocumentMutation,
      DeleteDriverLicenseDocumentMutationVariables
    >({
      mutation: DeleteDriverLicenseDocumentDocument,
      variables: {
        input,
      },
    });

    if (!result.data || result.data.deleteDocument.__typename === 'DeleteDocumentProblem') {
      return Promise.reject({
        reason: DeleteDocumentProblemReason.Internal,
      });
    }

    return 'ok';
  },
);

export const getDriverLicenseDataFx = createEffect<
  DocumentId,
  DriverLicenseDocument | null,
  GetDriverLicenseError
>(async (input) => {
  if (input.id) {
    const apollo = getOrCreateClient({});

    const result = await apollo.query<
      GetDriverLicenseByIdQuery,
      GetDriverLicenseByIdQueryVariables
    >({
      query: GetDriverLicenseByIdDocument,
      variables: {
        id: input.id,
      },
    });

    if (
      result.data.document.__typename === 'DriverLicenseDocument' &&
      result.data.document.type === DocumentType.DriverLicense
    ) {
      return result.data.document;
    }

    return Promise.reject({
      reason: DocumentByIdProblemReason.NotFound,
    });
  }

  return null;
});

export const uploadImagesFx = createEffect(async (input: UploadImagesFxInput) => {
  const { images, formData } = input;

  if (images) {
    const responses = await uploadImages<DocumentImage>(images, formData.id ?? '', false);

    const uploadedImages = responses.map((response) => response.data);

    return uploadedImages;
  }

  return [];
});

export const uploadAndSaveImagesFx = createEffect<
  UploadAndSaveImagesFxInput,
  DriverLicenseDocument
>(async (input) => {
  const { images: files, id } = input;

  await uploadImages<Document>(files, id, true);

  const apollo = getOrCreateClient({});

  const result = await apollo.query<GetDriverLicenseByIdQuery, GetDriverLicenseByIdQueryVariables>({
    query: GetDriverLicenseByIdDocument,
    fetchPolicy: 'network-only',
    variables: {
      id,
    },
  });

  if (result.data.document.__typename !== 'DriverLicenseDocument') {
    return Promise.reject({});
  }

  return result.data.document;
});

export const deleteImageFx = createEffect<
  DeleteImageFxInput,
  DriverLicenseDocument,
  { response: any }
>(async (input) => {
  const { document, imageId } = input;

  const images = document.images.filter((image) => image.id !== imageId);

  const response = await updateDriverLicense({
    ...document,
    images,
  });

  if (
    response.status === 'error' ||
    !response.data ||
    response.data.__typename !== 'DriverLicenseDocument'
  ) {
    return Promise.reject({ reason: response.reason });
  }

  return response.data;
});

export const temporaryDeleteImageFx = createEffect((input: TemporaryDeleteImageFxInput) => {
  const { images, imageId } = input;
  const newImages = images.filter((image) => image.id !== imageId);

  return newImages;
});

export const shareDocumentTextFx = createEffect<Field[], void, void>(async (fields) => {
  // Надо оборачивать в трайкетч, иначе отмена шаринга бросает исключение
  try {
    await navigator.share({
      title: 'ВУ',
      text: fields.reduce((acc, value) => acc + `${value.label}: ${value.text}\n`, ''),
    });
  } catch (e) {}
});

export const shareImagesFx = createEffect(shareImages);

export const shareImageFx = createEffect<DocumentImage, void, void>(async (image) => {
  return shareImages([image]);
});

async function addDriverLicense(document: Omit<DriverLicenseDocument, 'id' | 'section' | 'type'>) {
  const apollo = getOrCreateClient({});

  const images: DocumentImageInput[] = document.images.map((image) => ({
    id: image.id,
    originalHeight: image.originalHeight,
    originalUri: image.originalUri,
    originalUrl: image.originalUrl,
    originalWidth: image.originalWidth,
    previewHeight: image.previewHeight,
    previewUri: image.previewUri,
    previewUrl: image.previewUrl,
    previewWidth: image.previewWidth,
  }));

  const result = await apollo.mutate<AddDriverLicenseMutation, AddDriverLicenseMutationVariables>({
    mutation: AddDriverLicenseDocument,
    variables: {
      input: {
        type: DocumentType.DriverLicense,
        driverLicenseInput: {
          images,
          lastName: document.lastName,
          lastNameLat: document.lastNameLat,
          firstName: document.firstName,
          firstNameLat: document.firstNameLat,
          middleName: document.middleName,
          middleNameLat: document.middleNameLat,
          birthDate: document.birthDate,
          birthPlace: document.birthPlace,
          birthPlaceLat: document.birthPlaceLat,
          issueDate: document.issueDate,
          endDate: document.endDate,
          issuedBy: document.issuedBy,
          issuedByLat: document.issuedByLat,
          docNumber: document.docNumber,
          issuedCity: document.issuedCity,
          issuedCityLat: document.issuedCityLat,
          category: document.category,
          additionalInformation: document.additionalInformation,
        },
      },
    },
  });

  if (!result.data || result.data.addDocument.__typename === 'AddDocumentProblem') {
    return {
      status: 'error',
      reason: AddDocumentProblemReason.Internal,
    };
  }

  return {
    status: 'ok',
  };
}

async function updateDriverLicense(document: Omit<DriverLicenseDocument, 'section' | 'type'>) {
  const apollo = getOrCreateClient({});

  const images: DocumentImageInput[] = document.images.map((image) => ({
    id: image.id,
    originalHeight: image.originalHeight,
    originalUri: image.originalUri,
    originalUrl: image.originalUrl,
    originalWidth: image.originalWidth,
    previewHeight: image.previewHeight,
    previewUri: image.previewUri,
    previewUrl: image.previewUrl,
    previewWidth: image.previewWidth,
  }));

  const result = await apollo.mutate<
    UpdateDriverLicenseMutation,
    UpdateDriverLicenseMutationVariables
  >({
    mutation: UpdateDriverLicenseDocument,
    variables: {
      input: {
        type: DocumentType.DriverLicense,
        driverLicenseInput: {
          id: document.id,
          verificationStatus: document.verificationStatus,
          images,
          lastName: document.lastName,
          lastNameLat: document.lastNameLat,
          firstName: document.firstName,
          firstNameLat: document.firstNameLat,
          middleName: document.middleName,
          middleNameLat: document.middleNameLat,
          birthDate: document.birthDate,
          birthPlace: document.birthPlace,
          birthPlaceLat: document.birthPlaceLat,
          issueDate: document.issueDate,
          endDate: document.endDate,
          issuedBy: document.issuedBy,
          issuedByLat: document.issuedByLat,
          docNumber: document.docNumber,
          issuedCity: document.issuedCity,
          issuedCityLat: document.issuedCityLat,
          category: document.category,
          additionalInformation: document.additionalInformation,
        },
      },
      id: document.id,
    },
  });

  if (!result.data || result.data.updateDocument.__typename === 'UpdateDocumentProblem') {
    return {
      status: 'error',
      reason: UpdateDocumentProblemReason.Internal,
    };
  }

  return {
    status: 'ok',
    data:
      result.data.updateDocument.__typename === 'UpdateDocumentPayload' &&
      result.data.updateDocument.query.document,
  };
}

async function uploadImage<T>(file: File, id?: string, save = true) {
  const form = new FormData();

  form.append('file', file);
  form.append('type', DocumentType.DriverLicense);
  form.append('save', save ? 'true' : 'false');

  if (id) {
    form.append('id', id);
  }

  const response = await request<T>({
    url: `/documents/upload`,
    method: 'POST',
    data: form,
  });

  return response;
}

async function uploadImages<T>(files: FileList, id: string, save = true) {
  const uploadRequests = Object.values(files).reduce<Promise<AxiosResponse<T, any>>[]>(
    (acc, file) => {
      if (file.size < MAX_FILE_SIZE_BYTES) {
        const uploadDocumentData = uploadImage<T>(file, id, save);

        acc.push(uploadDocumentData);
      }

      return acc;
    },
    [],
  );

  return await Promise.all(uploadRequests);
}

async function shareImages(images: DocumentImage[]) {
  const files: File[] = [];

  for (let i = 0; i < images.length; i++) {
    const { originalUrl } = images[i];
    const path = new URL(originalUrl).pathname;
    const response = await request({
      url: 'avatars-proxy',
      params: { path },
      responseType: 'arraybuffer',
    });

    const type = response.headers['content-type'];
    const [_, extension] = type.split('/');

    files.push(new File([new Uint8Array(response.data)], `image${i + 1}.${extension}`, { type }));
  }

  // Надо оборачивать в трайкетч, иначе отмена шаринга бросает исключение
  try {
    await navigator.share({
      files,
    });
  } catch (e) {}
}
