import { attach, createEvent, createStore, sample } from 'effector';
import { createGate } from 'effector-react';

import { DocumentImage, DriverLicenseDocument } from '@client/shared/api/graphql';
import { createForm, rules } from '@client/shared/libs/effector-forms';
import { createDialogApi } from '@client/shared/libs/effector-helpers/dialog';

import { MAX_FILE_SIZE_BYTES } from '../../constants';
import { DocumentNotifier } from '../../notitier';
import * as api from './api';
import { DriverLicenseFormFields } from './interface';
import * as viewerModel from './viewer-model';

interface FormId {
  id?: string;
}

export const FormIdGate = createGate<FormId>();

// храним картинки в отдельном сторе, они могут удаляться, добавляться
// при сохранении документа отправляем картинки с этого стора
export const $images = createStore<DocumentImage[]>([]);
// Modal
export const dialog = createDialogApi('driver-license-form');
export const confirmationDialog = createDialogApi();
export const deleteImageDialog = createDialogApi();
export const imageViewerDialog = createDialogApi();

// Events
export const closeFormClicked = createEvent();
export const confirmCloseFormClicked = createEvent();
export const closeConfirmationDialogCliked = createEvent();

export const imageCliked = createEvent();
export const deleteImageClicked = createEvent();
export const confirmDeleteImageClicked = createEvent<{ imageId: string }>();
export const abortDeleteImageCliked = createEvent();

const resetImages = createEvent();

// Form
export const form = createForm<DriverLicenseFormFields>({
  fields: {
    images: {
      init: null as FileList | null,
    },
    id: {
      init: '',
    },
    lastName: {
      init: '',
    },
    lastNameLat: {
      init: '',
    },
    firstName: {
      init: '',
    },
    firstNameLat: {
      init: '',
    },
    middleName: {
      init: '',
    },
    middleNameLat: {
      init: '',
    },
    birthDate: {
      init: null as Date | null,
    },
    birthPlace: {
      init: '',
    },
    birthPlaceLat: {
      init: '',
    },
    issueDate: {
      init: null as Date | null,
    },
    endDate: {
      init: null as Date | null,
    },
    issuedBy: {
      init: '',
    },
    issuedByLat: {
      init: '',
    },
    docNumber: {
      init: '',
      rules: [rules.regexp(/^($|\d{10})$/)],
      filter: (value) => /^($|\d+)$/.test(value),
    },
    issuedCity: {
      init: '',
    },
    issuedCityLat: {
      init: '',
    },
    category: {
      init: '',
    },
    additionalInformation: {
      init: '',
    },
  },
  validateOn: ['change', 'blur', 'submit'],
});

// saveDriverLicenseFormDataFx
const saveDriverLicenseFormDataFx = attach({
  effect: api.saveDriverLicenseFormDataFx,
});

// getDriverLicenseDataFx
const getDriverLicenseDataFx = attach({
  effect: api.getDriverLicenseDataFx,
});

// deleteImageFx
const deleteImageFx = attach({
  effect: api.temporaryDeleteImageFx,
});

// uploadImagesFx
const uploadImagesFx = attach({
  effect: api.uploadImagesFx,
});

export const $isLoading = getDriverLicenseDataFx.pending;
export const $isPending = saveDriverLicenseFormDataFx.pending;
export const $isImagesUploading = uploadImagesFx.pending;
export const $isImageDeleting = deleteImageFx.pending;

// Обновления стора картинок после удаления / добавления
$images.on(uploadImagesFx.doneData, (currImages, addedImages) => [...currImages, ...addedImages]);
$images.on(deleteImageFx.doneData, (_, newImages) => newImages);
// Очистка стора
$images.reset([resetImages]);

// Получение данных по id документа
sample({
  clock: FormIdGate.state,
  target: [form.reset, resetImages, getDriverLicenseDataFx],
});

sample({
  clock: getDriverLicenseDataFx.failData,
  target: [dialog.hide, DocumentNotifier.notFound],
});

// Сохраняем данные в форму
sample({
  clock: getDriverLicenseDataFx.doneData,
  filter(document): document is DriverLicenseDocument {
    return (
      document !== null && document.verificationStatus !== 'full' && dialog.$isVisible.getState()
    );
  },
  fn: (document: DriverLicenseDocument) => {
    return {
      id: document.id,
      lastName: document.lastName ?? '',
      lastNameLat: document.lastNameLat ?? '',
      firstName: document.firstName ?? '',
      firstNameLat: document.firstNameLat ?? '',
      middleName: document.middleName ?? '',
      middleNameLat: document.middleNameLat ?? '',
      birthDate: document?.birthDate ? new Date(document.birthDate) : null,
      birthPlace: document.birthPlace ?? '',
      birthPlaceLat: document.birthPlaceLat ?? '',
      issueDate: document.issueDate ? new Date(document.issueDate) : null,
      endDate: document.endDate ? new Date(document.endDate) : null,
      issuedBy: document.issuedBy ?? '',
      issuedByLat: document.issuedByLat ?? '',
      docNumber: document.docNumber ?? '',
      issuedCity: document.issuedCity ?? '',
      issuedCityLat: document.issuedCityLat ?? '',
      category: document.category ?? '',
      additionalInformation: document.additionalInformation ?? '',
    };
  },
  target: form.setForm,
});

// Не открываем форму, если документ нельзя редактировать
// verificationStatus === full
sample({
  clock: getDriverLicenseDataFx.doneData,
  filter(document): document is DriverLicenseDocument {
    return document?.verificationStatus === 'full';
  },
  target: [dialog.hide, DocumentNotifier.cannotEdit],
});

// Сохраняем начальные картинки в стор картинок
sample({
  clock: getDriverLicenseDataFx.doneData,
  filter(document): document is DriverLicenseDocument {
    return (
      document !== null && document.verificationStatus !== 'full' && dialog.$isVisible.getState()
    );
  },
  fn: (document: DriverLicenseDocument) => document.images,
  target: $images,
});

// Отправка данных
sample({
  clock: form.formValidated,
  source: { formData: form.$values, images: $images },
  target: saveDriverLicenseFormDataFx,
});

// если это новый документ (нет id),
// то показываем снек об успешном добавлении докумнета
sample({
  clock: saveDriverLicenseFormDataFx.doneData,
  source: form.$values,
  filter: (form) => !form.id,
  target: [DocumentNotifier.documentAdded.prepend(() => 'ВУ'), dialog.hide],
});

// иначе закрываем форму и открываем карточку документа
sample({
  clock: saveDriverLicenseFormDataFx.doneData,
  source: form.$values,
  filter: (form) => Boolean(form.id),
  target: dialog.hide,
});

sample({
  clock: dialog.hide,
  source: form.$values.map((formData) => ({ id: formData.id })),
  filter: (form) => Boolean(form.id),
  target: [viewerModel.dialog.setParams, viewerModel.dialog.show],
});

// Просмотр картинок
sample({
  clock: imageCliked,
  target: imageViewerDialog.show,
});

// Загрузка картинок
sample({
  clock: form.fields.images.changed,
  source: { images: form.fields.images.$value, formData: form.$values },
  target: uploadImagesFx,
});

// Обработка больших картинок
sample({
  clock: form.fields.images.changed,
  filter: (images) => {
    if (images) {
      return Object.values(images).some((image) => image.size > MAX_FILE_SIZE_BYTES);
    }

    return false;
  },
  target: DocumentNotifier.maxFileSize,
});

// Удаление картинки
sample({
  clock: deleteImageClicked,
  target: deleteImageDialog.show,
});

sample({
  clock: abortDeleteImageCliked,
  target: deleteImageDialog.hide,
});

sample({
  clock: confirmDeleteImageClicked,
  source: $images,
  fn: (images, image) => ({ images, imageId: image.imageId }),
  target: deleteImageFx,
});

sample({
  clock: deleteImageFx.doneData,
  target: [deleteImageDialog.hide, imageViewerDialog.hide],
});

// Захотели выйти и есть данные - спрашиваем
sample({
  source: closeFormClicked,
  filter: form.$isTouched,
  target: confirmationDialog.show,
});

// Захотели выйти и нет данных - выходим
sample({
  source: closeFormClicked,
  filter: form.$isTouched.map(($isTouched) => !$isTouched),
  target: [confirmationDialog.hide, dialog.hide],
});

// Подтвердили выход - скрываем всё
sample({
  clock: confirmCloseFormClicked,
  target: [confirmationDialog.hide, dialog.hide, resetImages],
});

// Передумали выходить - остаемся на формe
sample({
  clock: closeConfirmationDialogCliked,
  target: confirmationDialog.hide,
});

// Очищаем состояние после закрытия формы
sample({
  clock: dialog.hide,
  target: [form.reset, resetImages],
});
