import { useEvent, useGate, useStore } from 'effector-react';
import { ChangeEventHandler, VFC, useCallback, useState } from 'react';

import { CloseFormDialog, DocumentForm } from '@client/entities/document-form';
import { DeleteImageDialog, DocumentImageList } from '@client/entities/document-image-list';
import { DocumentGender } from '@client/shared/api/graphql';
import { useForm } from '@client/shared/libs/effector-forms';
import { useQueryParams } from '@client/shared/libs/router';
import { DateField, RadioButton, Text, TextField } from '@client/shared/ui-kit';

import { InnFormSkeleton } from './InnFormSkeleton';
import * as model from './form-model';
import { FieldType } from './interface';

import styles from './InnForm.module.css';

export interface InnFormQueryParams {
  field?: FieldType;
  id?: string;
}

export const InnForm: VFC = () => {
  const form = useForm(model.form);
  const queryParams = useQueryParams<InnFormQueryParams>();

  const images = useStore(model.$images);

  const isVisible = useStore(model.dialog.$isVisible);
  const isLoading = useStore(model.$isLoading);
  const isPending = useStore(model.$isPending);
  const isImagesUploading = useStore(model.$isImagesUploading);
  const isImageDeleting = useStore(model.$isImageDeleting);
  const isImageViewerVisible = useStore(model.imageViewerDialog.$isVisible);
  const isConfirmationDialogVisible = useStore(model.confirmationDialog.$isVisible);
  const isDeleteImageDialogVisible = useStore(model.deleteImageDialog.$isVisible);

  const handleOnSubmit = useEvent(form.submit);
  const handleOnClose = useEvent(model.closeFormClicked);
  const handleOnCloseConfirm = useEvent(model.confirmCloseFormClicked);
  const handleOnCloseConfirmation = useEvent(model.closeConfirmationDialogCliked);

  const handleOnImageClicked = useEvent(model.imageCliked);
  const handleOnImageViewerClose = useEvent(model.imageViewerDialog.hide);
  const handleOnDeleteImage = useEvent(model.deleteImageClicked);
  const handleOnDeleteImageConfirm = useEvent(model.confirmDeleteImageClicked);
  const handleOnDeleteImageAbort = useEvent(model.abortDeleteImageCliked);

  useGate(model.FormIdGate, {
    id: isVisible ? queryParams.id ?? '' : '',
  });

  const handleOnImagesUpload = useCallback(
    (images: FileList | null) => {
      if (images && images.length > 0) {
        form.fields.images.onChange(images);
      }
    },
    [form.fields.images],
  );

  const [activeImageId, setActiveImageId] = useState<string | null>(null);

  const deleteImage = useCallback(() => {
    if (activeImageId) {
      handleOnDeleteImageConfirm({
        imageId: activeImageId,
      });
    }
  }, [activeImageId, handleOnDeleteImageConfirm]);

  const onGenderChangeTab: ChangeEventHandler<HTMLInputElement> = useCallback(
    (e) => {
      form.fields.gender.onChange(e.target.value as DocumentGender);
    },
    [form.fields.gender],
  );

  if (!isVisible) {
    return null;
  }

  if (document === null || isLoading) {
    return <InnFormSkeleton />;
  }

  return (
    <>
      <DocumentForm
        isVisible={isVisible}
        isDisabled={isPending || !(form.isDirty && form.isValid)}
        onClose={handleOnClose}
        onSubmit={handleOnSubmit}
      >
        <DocumentForm.Title text="ИНН" />
        <DocumentForm.Preview>
          <DocumentImageList
            onImageClick={handleOnImageClicked}
            onImageViewerClose={handleOnImageViewerClose}
            isImageViewerVisible={isImageViewerVisible}
            onDelete={(id) => {
              setActiveImageId(id);
              handleOnDeleteImage();
            }}
            isUploading={isImagesUploading}
            onUpload={handleOnImagesUpload}
            isDownloadDisabled
            images={images}
            target="upload"
          />
        </DocumentForm.Preview>
        <DocumentForm.Section>
          <TextField
            autoFocus={queryParams.field === 'docNumber'}
            label="ИНН *"
            inputMode="decimal"
            error={form.fields.docNumber.errorText({
              required: 'Введите цифры без тире и пробелов (12 цифр)',
            })}
            value={form.fields.docNumber.value}
            onBlur={() => form.fields.docNumber.onBlur()}
            onChangeValue={form.fields.docNumber.onChange}
          />
          <DocumentForm.MultiField>
            <TextField
              autoFocus={queryParams.field === 'lastName'}
              label="Фамилия, имя и отчество"
              placeholder="Фамилия"
              value={form.fields.lastName.value}
              onChangeValue={form.fields.lastName.onChange}
            />
            <TextField
              autoFocus={queryParams.field === 'firstName'}
              placeholder="Имя"
              value={form.fields.firstName.value}
              onChangeValue={form.fields.firstName.onChange}
            />
            <TextField
              autoFocus={queryParams.field === 'middleName'}
              placeholder="Отчество"
              value={form.fields.middleName.value}
              onChangeValue={form.fields.middleName.onChange}
            />
          </DocumentForm.MultiField>
          <DateField
            size="xl"
            label="Дата рождения"
            autoFocus={queryParams.field === 'birthDate'}
            value={form.fields.birthDate.value}
            onChangeValue={form.fields.birthDate.onChange}
          />
          <div className={styles.fieldSection}>
            <Text variant="text-m-long">Пол</Text>
            <RadioButton
              className={styles.tabs}
              size="l"
              variant="outline"
              value={form.fields.gender.value?.toString() ?? undefined}
              options={[
                { value: DocumentGender.Male.toString(), children: 'Мужской' },
                { value: DocumentGender.Female.toString(), children: 'Женский' },
              ]}
              onChange={onGenderChangeTab}
            />
          </div>
        </DocumentForm.Section>
        <DocumentForm.Section>
          <TextField
            autoFocus={queryParams.field === 'issuedBy'}
            label="Орган, выдавший документ"
            value={form.fields.issuedBy.value}
            onChangeValue={form.fields.issuedBy.onChange}
          />
          <DateField
            size="xl"
            label="Дата выдачи"
            autoFocus={queryParams.field === 'issueDate'}
            value={form.fields.issueDate.value}
            onChangeValue={form.fields.issueDate.onChange}
          />
        </DocumentForm.Section>
      </DocumentForm>
      <CloseFormDialog
        isVisible={isConfirmationDialogVisible}
        onCloseDialog={handleOnCloseConfirmation}
        onContinueEditing={handleOnCloseConfirmation}
        onCloseForm={handleOnCloseConfirm}
      />
      <DeleteImageDialog
        isVisible={isDeleteImageDialogVisible}
        disabled={isImageDeleting}
        onConfirm={deleteImage}
        onAbort={handleOnDeleteImageAbort}
      />
    </>
  );
};
