import Axios, { AxiosResponse } from 'axios';
import { createEffect } from 'effector';

import {
  AddDocumentProblemReason,
  AddInnDocument,
  AddInnMutation,
  AddInnMutationVariables,
  DeleteDocumentInput,
  DeleteDocumentProblemReason,
  DeleteInnDocumentDocument,
  DeleteInnDocumentMutation,
  DeleteInnDocumentMutationVariables,
  DocumentByIdProblemReason,
  DocumentImage,
  DocumentImageInput,
  DocumentType,
  GetInnByIdDocument,
  GetInnByIdQuery,
  GetInnByIdQueryVariables,
  InnDocument,
  UpdateDocumentProblemReason,
  UpdateInnDocument,
  UpdateInnMutation,
  UpdateInnMutationVariables,
} from '@client/shared/api/graphql';
import { getOrCreateClient } from '@client/shared/libs/apollo';

import { MAX_FILE_SIZE_BYTES } from '../../../constants';
import { formatDateToISO } from '../../../libs/date';
import { InnFormFields } from '../interface';
import { Field } from '../types/field';

const { request } = Axios.create({
  baseURL: '/personal/api/',
  // TODO: вот тут вопрос
  // сколько ставить
  // а еще мб как-нибудь ограничить рамер файла?
  timeout: 10000,
});

export interface DocumentId {
  id?: string;
}

export interface SaveInnFormDataFxInput {
  formData: InnFormFields;
  images: DocumentImage[];
}

export interface UploadImagesFxInput {
  formData: InnFormFields;
  images: FileList | null;
}

export interface UploadAndSaveImagesFxInput {
  id: string;
  images: FileList;
}

export interface DeleteImageFxInput {
  document: InnDocument;
  imageId: string;
}

interface DeleteDocumentError {
  reason: DeleteDocumentProblemReason;
}

interface GetInnError {
  reason: DocumentByIdProblemReason;
}

interface TemporaryDeleteImageFxInput {
  images: DocumentImage[];
  imageId: string;
}

export const saveInnFormDataFx = createEffect(async (input: SaveInnFormDataFxInput) => {
  const { formData, images } = input;

  if (formData.id) {
    const response = await updateInn({
      id: formData.id,
      images,
      docNumber: formData.docNumber,
      lastName: formData.lastName,
      firstName: formData.firstName,
      middleName: formData.middleName,
      birthDate: formatDateToISO(formData.birthDate),
      gender: formData.gender,
      issueDate: formatDateToISO(formData.issueDate),
      issuedBy: formData.issuedBy,
    });

    if (response.status === 'error') {
      return Promise.reject({ reason: response.reason });
    }
  } else {
    const response = await addInn({
      images,
      docNumber: formData.docNumber,
      lastName: formData.lastName,
      firstName: formData.firstName,
      middleName: formData.middleName,
      birthDate: formatDateToISO(formData.birthDate),
      gender: formData.gender,
      issueDate: formatDateToISO(formData.issueDate),
      issuedBy: formData.issuedBy,
    });

    if (response.status === 'error') {
      return Promise.reject({ reason: response.reason });
    }
  }
});

export const deleteInnFx = createEffect<DeleteDocumentInput, string, DeleteDocumentError>(
  async (input) => {
    const apollo = getOrCreateClient({});

    const result = await apollo.mutate<
      DeleteInnDocumentMutation,
      DeleteInnDocumentMutationVariables
    >({
      mutation: DeleteInnDocumentDocument,
      variables: {
        input,
      },
    });

    if (!result.data || result.data.deleteDocument.__typename === 'DeleteDocumentProblem') {
      return Promise.reject({
        reason: DeleteDocumentProblemReason.Internal,
      });
    }

    return 'ok';
  },
);

export const getInnDataFx = createEffect<DocumentId, InnDocument | null, GetInnError>(
  async (input) => {
    if (input.id) {
      const apollo = getOrCreateClient({});

      const result = await apollo.query<GetInnByIdQuery, GetInnByIdQueryVariables>({
        query: GetInnByIdDocument,
        variables: {
          id: input.id,
        },
      });

      if (
        result.data.document.__typename === 'InnDocument' &&
        result.data.document.type === DocumentType.Inn
      ) {
        return result.data.document;
      }

      return Promise.reject({
        reason: DocumentByIdProblemReason.NotFound,
      });
    }

    return null;
  },
);

export const uploadImagesFx = createEffect(async ({ images, formData }: UploadImagesFxInput) => {
  if (images) {
    const responses = await uploadImages<DocumentImage>(images, formData.id ?? '', false);

    const uploadedImages = responses.map((response) => response.data);

    return uploadedImages;
  }

  return [];
});

export const uploadAndSaveImagesFx = createEffect<UploadAndSaveImagesFxInput, InnDocument>(
  async (input) => {
    const { images: files, id } = input;

    await uploadImages<Document>(files, id, true);

    const apollo = getOrCreateClient({});

    const result = await apollo.query<GetInnByIdQuery, GetInnByIdQueryVariables>({
      query: GetInnByIdDocument,
      fetchPolicy: 'network-only',
      variables: {
        id,
      },
    });

    if (result.data.document.__typename !== 'InnDocument') {
      return Promise.reject({});
    }

    return result.data.document;
  },
);

export const deleteImageFx = createEffect<DeleteImageFxInput, InnDocument, { response: any }>(
  async (input) => {
    const { document, imageId } = input;

    const images = document.images.filter((image) => image.id !== imageId);

    const response = await updateInn({
      ...document,
      images,
    });

    if (
      response.status === 'error' ||
      !response.data ||
      response.data.__typename !== 'InnDocument'
    ) {
      return Promise.reject({ reason: response.reason });
    }

    return response.data;
  },
);

export const temporaryDeleteImageFx = createEffect((input: TemporaryDeleteImageFxInput) => {
  const { images, imageId } = input;
  const newImages = images.filter((image) => image.id !== imageId);

  return newImages;
});

export const shareDocumentTextFx = createEffect<Field[], void, void>(async (fields) => {
  // Надо оборачивать в трайкетч, иначе отмена шаринга бросает исключение
  try {
    await navigator.share({
      title: 'ИНН',
      text: fields.reduce((acc, value) => acc + `${value.label}: ${value.text}\n`, ''),
    });
  } catch (e) {}
});

export const shareImagesFx = createEffect(shareImages);

export const shareImageFx = createEffect<DocumentImage, void, void>(async (image) => {
  return shareImages([image]);
});

async function addInn(document: Omit<InnDocument, 'id' | 'section' | 'type'>) {
  const apollo = getOrCreateClient({});

  const images: DocumentImageInput[] = document.images.map((image) => ({
    id: image.id,
    originalHeight: image.originalHeight,
    originalUri: image.originalUri,
    originalUrl: image.originalUrl,
    originalWidth: image.originalWidth,
    previewHeight: image.previewHeight,
    previewUri: image.previewUri,
    previewUrl: image.previewUrl,
    previewWidth: image.previewWidth,
  }));

  const result = await apollo.mutate<AddInnMutation, AddInnMutationVariables>({
    mutation: AddInnDocument,
    variables: {
      input: {
        type: DocumentType.Inn,
        innInput: {
          images,
          docNumber: document.docNumber,
          firstName: document.firstName,
          lastName: document.lastName,
          middleName: document.middleName,
          birthDate: document.birthDate,
          gender: document.gender,
          issuedBy: document.issuedBy,
          issueDate: document.issueDate,
        },
      },
    },
  });

  if (!result.data || result.data.addDocument.__typename === 'AddDocumentProblem') {
    return {
      status: 'error',
      reason: AddDocumentProblemReason.Internal,
    };
  }

  return {
    status: 'ok',
  };
}

async function updateInn(document: Omit<InnDocument, 'section' | 'type'>) {
  const apollo = getOrCreateClient({});

  const images: DocumentImageInput[] = document.images.map((image) => ({
    id: image.id,
    originalHeight: image.originalHeight,
    originalUri: image.originalUri,
    originalUrl: image.originalUrl,
    originalWidth: image.originalWidth,
    previewHeight: image.previewHeight,
    previewUri: image.previewUri,
    previewUrl: image.previewUrl,
    previewWidth: image.previewWidth,
  }));

  const result = await apollo.mutate<UpdateInnMutation, UpdateInnMutationVariables>({
    mutation: UpdateInnDocument,
    variables: {
      input: {
        type: DocumentType.Inn,
        innInput: {
          id: document.id,
          verificationStatus: document.verificationStatus,
          images,
          docNumber: document.docNumber,
          firstName: document.firstName,
          lastName: document.lastName,
          middleName: document.middleName,
          birthDate: document.birthDate,
          gender: document.gender,
          issuedBy: document.issuedBy,
          issueDate: document.issueDate,
        },
      },
      id: document.id,
    },
  });

  if (!result.data || result.data.updateDocument.__typename === 'UpdateDocumentProblem') {
    return {
      status: 'error',
      reason: UpdateDocumentProblemReason.Internal,
    };
  }

  return {
    status: 'ok',
    data:
      result.data.updateDocument.__typename === 'UpdateDocumentPayload' &&
      result.data.updateDocument.query.document,
  };
}

async function uploadImage<T>(file: File, id?: string, save = true) {
  const form = new FormData();

  form.append('file', file);
  form.append('type', DocumentType.NationalId);
  form.append('save', save ? 'true' : 'false');

  if (id) {
    form.append('id', id);
  }

  const response = await request<T>({
    url: `/documents/upload`,
    method: 'POST',
    data: form,
  });

  return response;
}

async function uploadImages<T>(files: FileList, id: string, save = true) {
  const uploadRequests = Object.values(files).reduce<Promise<AxiosResponse<T, any>>[]>(
    (acc, file) => {
      if (file.size < MAX_FILE_SIZE_BYTES) {
        const uploadDocumentData = uploadImage<T>(file, id, save);

        acc.push(uploadDocumentData);
      }

      return acc;
    },
    [],
  );

  return await Promise.all(uploadRequests);
}

async function shareImages(images: DocumentImage[]) {
  const files: File[] = [];

  for (let i = 0; i < images.length; i++) {
    const { originalUrl } = images[i];
    const path = new URL(originalUrl).pathname;
    const response = await request({
      url: 'avatars-proxy',
      params: { path },
      responseType: 'arraybuffer',
    });

    const type = response.headers['content-type'];
    const [_, extension] = type.split('/');

    files.push(new File([new Uint8Array(response.data)], `image${i + 1}.${extension}`, { type }));
  }

  // Надо оборачивать в трайкетч, иначе отмена шаринга бросает исключение
  try {
    await navigator.share({
      files,
    });
  } catch (e) {}
}
