import { useEvent, useGate, useStore } from 'effector-react';
import { VFC, useCallback, useState } from 'react';

import { DeleteImageDialog, DocumentImageList } from '@client/entities/document-image-list';
import {
  DeleteDocumentDialog,
  DocumentViewer,
  DocumentViewerSkeleton,
  ShareDialog,
} from '@client/entities/document-viewer';
import { useDevice } from '@client/shared/hooks';
import { PlaneIcon } from '@client/shared/icons';
import { useForm } from '@client/shared/libs/effector-forms';
import { useQueryParams } from '@client/shared/libs/router';
import { Text } from '@client/shared/ui-kit';

import { getFullName } from '../../libs/full-name';
import * as deleteDocumentModel from './delete-document-model';
import * as formModel from './form-model';
import * as imagesModel from './images-model';
import { FieldType } from './interface';
import { getBlankFields } from './libs/blank-fields';
import { splitFieldsIntoSections } from './libs/field-sections';
import * as shareModel from './share-model';
import * as viewerModel from './viewer-model';

interface QueryParams {
  id: string;
}

export const InternationalIdViewer: VFC = () => {
  const queryParams = useQueryParams<QueryParams>();

  const imagesForm = useForm(imagesModel.imagesForm);
  const { canShare, isTouch } = useDevice();

  const document = useStore(viewerModel.$document);
  const isVisible = useStore(viewerModel.dialog.$isVisible);
  const isLoading = useStore(viewerModel.$isLoading);
  const isImageViewerVisible = useStore(imagesModel.imageViewerDialog.$isVisible);
  const isDocumentDeleting = useStore(deleteDocumentModel.$isInternationalIdDeleting);
  const isImagesUploading = useStore(imagesModel.$isImagesUploading);
  const isImageDeleting = useStore(imagesModel.$isImageDeleting);
  const isDeleteDocumentDialogVisible = useStore(deleteDocumentModel.dialog.$isVisible);
  const isDeleteImageDialogVisible = useStore(imagesModel.deleteImageDialog.$isVisible);
  const isShareDocumentDialogVisible = useStore(shareModel.dialog.$isVisible);
  const shareData = useStore(shareModel.$shareData);

  const handleOnClose = useEvent(viewerModel.dialog.hide);
  const handleOnDelete = useEvent(deleteDocumentModel.deleteDocumentClicked);
  const handleOnShare = useEvent(shareModel.shareDocumentClicked);
  const handleOnDeleteDocumentConfirm = useEvent(deleteDocumentModel.confirmDeleteDocumentClicked);
  const handleOnDeleteDocumentAbort = useEvent(deleteDocumentModel.abortDeleteDocumentClicked);
  const handleOnImageClicked = useEvent(imagesModel.imageCliked);
  const handleOnImageViewerClose = useEvent(imagesModel.imageViewerDialog.hide);
  const handleOnImageDelete = useEvent(imagesModel.deleteImageClicked);
  const handleOnDeleteImageConfirm = useEvent(imagesModel.confirmDeleteImageClicked);
  const handleOnDeleteImageAbort = useEvent(imagesModel.abortDeleteImageCliked);
  const handleOnShareDocumentAbort = useEvent(shareModel.abortShareDocumentClicked);
  const handleDocumentTextShare = useEvent(shareModel.shareDocumentText);
  const handleDocumentImagesShare = useEvent(shareModel.shareDocumentImages);

  const [activeImageId, setActiveImageId] = useState<string | null>(null);

  const deleteImage = useCallback(() => {
    if (activeImageId) {
      handleOnDeleteImageConfirm({
        imageId: activeImageId,
      });
    }
  }, [activeImageId, handleOnDeleteImageConfirm]);

  const handleOnImagesUpload = useCallback(
    (images: FileList | null) => {
      if (document && images && images.length > 0) {
        imagesForm.fields.id.onChange(document.id);
        imagesForm.fields.images.onChange(images);
      }
    },
    [imagesForm.fields.images, document, imagesForm.fields.id],
  );

  const handleOnEdit = useCallback(() => {
    if (document) {
      formModel.dialog.setParams({ id: document.id });
      formModel.dialog.show();
    }
  }, [document]);

  const generateOnFill = useCallback(
    (field: FieldType) => () => {
      if (document) {
        formModel.dialog.setParams({ id: document.id, field });
        formModel.dialog.show();
      }
    },
    [document],
  );

  useGate(viewerModel.DocumentIdGate, {
    id: isVisible ? queryParams.id ?? '' : '',
  });

  if (!isVisible) {
    return null;
  }

  if (document === null || isLoading) {
    return (
      <DocumentViewerSkeleton color="red" icon={<PlaneIcon size={32} />} onClose={handleOnClose} />
    );
  }

  const sections = splitFieldsIntoSections(document);
  const blankFields = getBlankFields({
    document,
    generateOnPress: generateOnFill,
  });
  const name = getFullName({
    firstName: document.firstName,
    lastName: document.lastName,
    withPatronymic: false,
  });

  return (
    <>
      <DocumentViewer color="red" isVisible={isVisible} onClose={handleOnClose}>
        <DocumentViewer.Icon color="red">
          <PlaneIcon size={32} />
        </DocumentViewer.Icon>
        <DocumentViewer.Title>
          <Text as="h2" variant="heading-l" overflow="ellipsis">
            {name || 'Загранпаспорт'}
          </Text>
        </DocumentViewer.Title>
        {name && (
          <DocumentViewer.Subtitle>
            <Text variant="text-m" color="secondary">
              Загранпаспорт
            </Text>
          </DocumentViewer.Subtitle>
        )}
        <DocumentViewer.Preview>
          <DocumentImageList
            onImageClick={handleOnImageClicked}
            onImageViewerClose={handleOnImageViewerClose}
            isImageViewerVisible={isImageViewerVisible}
            onDelete={(id) => {
              setActiveImageId(id);
              handleOnImageDelete();
            }}
            isUploading={isImagesUploading}
            onUpload={handleOnImagesUpload}
            images={document.images}
            target="view"
          />
        </DocumentViewer.Preview>
        {sections.map((section) => (
          <DocumentViewer.Section key={section.name}>
            {section.fields.map((field) => (
              <DocumentViewer.FieldItem key={field.label} text={field.text} label={field.label} />
            ))}
          </DocumentViewer.Section>
        ))}
        {blankFields.length > 0 && <DocumentViewer.Blanks elements={blankFields} />}
        <DocumentViewer.DeleteAction onPress={handleOnDelete} />
        {document.verificationStatus !== 'full' && (
          <DocumentViewer.EditAction onPress={handleOnEdit} />
        )}
        {isTouch && canShare && (shareData.images.length > 0 || shareData.fields.length > 0) && (
          <DocumentViewer.ShareAction onPress={handleOnShare} />
        )}
      </DocumentViewer>
      <DeleteDocumentDialog
        isVisible={isDeleteDocumentDialogVisible}
        disabled={isDocumentDeleting}
        name="Загранпаспорт"
        onConfirm={handleOnDeleteDocumentConfirm}
        onAbort={handleOnDeleteDocumentAbort}
      />
      <DeleteImageDialog
        isVisible={isDeleteImageDialogVisible}
        disabled={isImageDeleting}
        onConfirm={deleteImage}
        onAbort={handleOnDeleteImageAbort}
      />
      <ShareDialog
        isVisible={isShareDocumentDialogVisible}
        onAbort={handleOnShareDocumentAbort}
        onShareText={handleDocumentTextShare}
        onShareImages={handleDocumentImagesShare}
        imagesQuantity={shareData.images.length}
      />
    </>
  );
};
