import Axios, { AxiosResponse } from 'axios';
import { createEffect } from 'effector';

import {
  AddDocumentProblemReason,
  AddInternationalIdDocument,
  AddInternationalIdMutation,
  AddInternationalIdMutationVariables,
  DeleteDocumentInput,
  DeleteDocumentProblemReason,
  DeleteInternationalIdDocumentDocument,
  DeleteInternationalIdDocumentMutation,
  DeleteInternationalIdDocumentMutationVariables,
  DocumentByIdProblemReason,
  DocumentImage,
  DocumentImageInput,
  DocumentType,
  GetInternationalIdByIdDocument,
  GetInternationalIdByIdQuery,
  GetInternationalIdByIdQueryVariables,
  InternationalIdDocument,
  UpdateDocumentProblemReason,
  UpdateInternationalIdDocument,
  UpdateInternationalIdMutation,
  UpdateInternationalIdMutationVariables,
} from '@client/shared/api/graphql';
import { getOrCreateClient } from '@client/shared/libs/apollo';

import { MAX_FILE_SIZE_BYTES } from '../../../constants';
import { formatDateToISO } from '../../../libs/date';
import { InternationalIdFormFields } from '../interface';
import { Field } from '../types/field';

const { request } = Axios.create({
  baseURL: '/personal/api/',
  // TODO: вот тут вопрос
  // сколько ставить
  // а еще мб как-нибудь ограничить рамер файла?
  timeout: 10000,
});

export interface DocumentId {
  id?: string;
}

export interface SaveInternationalIdFormDataFxInput {
  formData: InternationalIdFormFields;
  images: DocumentImage[];
}

export interface UploadImagesFxInput {
  formData: InternationalIdFormFields;
  images: FileList | null;
}

export interface UploadAndSaveImagesFxInput {
  id: string;
  images: FileList;
}

export interface SaveInternationalIdFormDataFxParams {
  formData: InternationalIdFormFields;
  document: InternationalIdDocument | null;
}

interface DeleteDocumentError {
  reason: DeleteDocumentProblemReason;
}

interface GetInternationalIdError {
  reason: DocumentByIdProblemReason;
}

interface temporaryDeleteImageFxInput {
  images: DocumentImage[];
  imageId: string;
}

export const saveInternationalIdFormDataFx = createEffect(
  async ({
    formData,
    images,
  }: {
    formData: InternationalIdFormFields;
    images: DocumentImage[];
  }) => {
    if (formData.id) {
      const response = await updateInternationalId({
        id: formData.id,
        images,
        docNumber: formData.docNumber,
        lastName: formData.lastName,
        lastNameLat: formData.lastNameLat,
        firstName: formData.firstName,
        firstNameLat: formData.firstNameLat,
        citizenship: formData.citizenship,
        citizenshipLat: formData.citizenshipLat,
        birthDate: formatDateToISO(formData.birthDate),
        gender: formData.gender,
        birthPlace: formData.birthPlace,
        birthPlaceLat: formData.birthPlaceLat,
        endDate: formatDateToISO(formData.endDate),
        issueDate: formatDateToISO(formData.issueDate),
        issuedBy: formData.issuedBy,
      });

      if (response.status === 'error') {
        return Promise.reject({ reason: response.reason });
      }
    } else {
      const response = await addInternationalId({
        images,
        docNumber: formData.docNumber,
        lastName: formData.lastName,
        lastNameLat: formData.lastNameLat,
        firstName: formData.firstName,
        firstNameLat: formData.firstNameLat,
        citizenship: formData.citizenship,
        citizenshipLat: formData.citizenshipLat,
        birthDate: formatDateToISO(formData.birthDate),
        gender: formData.gender,
        birthPlace: formData.birthPlace,
        birthPlaceLat: formData.birthPlaceLat,
        endDate: formatDateToISO(formData.endDate),
        issueDate: formatDateToISO(formData.issueDate),
        issuedBy: formData.issuedBy,
      });

      if (response.status === 'error') {
        return Promise.reject({ reason: response.reason });
      }
    }
  },
);

export const deleteInternationalIdFx = createEffect<
  DeleteDocumentInput,
  string,
  DeleteDocumentError
>(async (input) => {
  const apollo = getOrCreateClient({});

  const result = await apollo.mutate<
    DeleteInternationalIdDocumentMutation,
    DeleteInternationalIdDocumentMutationVariables
  >({
    mutation: DeleteInternationalIdDocumentDocument,
    variables: {
      input,
    },
  });

  if (!result.data || result.data.deleteDocument.__typename === 'DeleteDocumentProblem') {
    return Promise.reject({
      reason: DeleteDocumentProblemReason.Internal,
    });
  }

  return 'ok';
});

export const getInternationalIdDataFx = createEffect<
  DocumentId,
  InternationalIdDocument | null,
  GetInternationalIdError
>(async (input) => {
  if (input.id) {
    const apollo = getOrCreateClient({});

    const result = await apollo.query<
      GetInternationalIdByIdQuery,
      GetInternationalIdByIdQueryVariables
    >({
      query: GetInternationalIdByIdDocument,
      variables: {
        id: input.id,
      },
    });

    if (
      result.data.document.__typename === 'InternationalIdDocument' &&
      result.data.document.type === DocumentType.InternationalId
    ) {
      return result.data.document;
    }

    return Promise.reject({
      reason: DocumentByIdProblemReason.NotFound,
    });
  }

  return null;
});

export const uploadImagesFx = createEffect(async ({ images, formData }: UploadImagesFxInput) => {
  if (images) {
    const responses = await uploadImages<DocumentImage>(images, formData.id ?? '', false);

    const uploadedImages = responses.map((response) => response.data);

    return uploadedImages;
  }

  return [];
});

export const uploadAndSaveImagesFx = createEffect<
  UploadAndSaveImagesFxInput,
  InternationalIdDocument
>(async (input) => {
  const { images: files, id } = input;

  await uploadImages<Document>(files, id, true);

  const apollo = getOrCreateClient({});

  const result = await apollo.query<
    GetInternationalIdByIdQuery,
    GetInternationalIdByIdQueryVariables
  >({
    query: GetInternationalIdByIdDocument,
    fetchPolicy: 'network-only',
    variables: {
      id,
    },
  });

  if (result.data.document.__typename !== 'InternationalIdDocument') {
    return Promise.reject({});
  }

  return result.data.document;
});

export const deleteImageFx = createEffect<
  { document: InternationalIdDocument; imageId: string },
  InternationalIdDocument,
  { response: any }
>(async (input) => {
  const { document, imageId } = input;

  const images = document.images.filter((image) => image.id !== imageId);

  const response = await updateInternationalId({
    ...document,
    images,
  });

  if (
    response.status === 'error' ||
    !response.data ||
    response.data.__typename !== 'InternationalIdDocument'
  ) {
    return Promise.reject({ reason: response.reason });
  }

  return response.data;
});

export const temporaryDeleteImageFx = createEffect((input: temporaryDeleteImageFxInput) => {
  const { images, imageId } = input;
  const newImages = images.filter((image) => image.id !== imageId);

  return newImages;
});

export const shareDocumentTextFx = createEffect<Field[], void, void>(async (fields) => {
  // Надо оборачивать в трайкетч, иначе отмена шаринга бросает исключение
  try {
    await navigator.share({
      title: 'Загранпаспорт',
      text: fields.reduce((acc, value) => acc + `${value.label}: ${value.text}\n`, ''),
    });
  } catch (e) {}
});

export const shareImagesFx = createEffect(shareImages);

export const shareImageFx = createEffect<DocumentImage, void, void>(async (image) => {
  return shareImages([image]);
});

async function addInternationalId(
  document: Omit<InternationalIdDocument, 'id' | 'section' | 'type'>,
) {
  const apollo = getOrCreateClient({});

  const images: DocumentImageInput[] = document.images.map((image) => ({
    id: image.id,
    originalHeight: image.originalHeight,
    originalUri: image.originalUri,
    originalUrl: image.originalUrl,
    originalWidth: image.originalWidth,
    previewHeight: image.previewHeight,
    previewUri: image.previewUri,
    previewUrl: image.previewUrl,
    previewWidth: image.previewWidth,
  }));

  const result = await apollo.mutate<
    AddInternationalIdMutation,
    AddInternationalIdMutationVariables
  >({
    mutation: AddInternationalIdDocument,
    variables: {
      input: {
        type: DocumentType.InternationalId,
        internationalIdInput: {
          images,
          docNumber: document.docNumber,
          lastName: document.lastName,
          lastNameLat: document.lastNameLat,
          firstName: document.firstName,
          firstNameLat: document.firstNameLat,
          citizenship: document.citizenship,
          citizenshipLat: document.citizenshipLat,
          birthDate: document.birthDate,
          gender: document.gender,
          birthPlace: document.birthPlace,
          birthPlaceLat: document.birthPlaceLat,
          endDate: document.endDate,
          issueDate: document.issueDate,
          issuedBy: document.issuedBy,
        },
      },
    },
  });

  if (!result.data || result.data.addDocument.__typename === 'AddDocumentProblem') {
    return {
      status: 'error',
      reason: AddDocumentProblemReason.Internal,
    };
  }

  return {
    status: 'ok',
  };
}

async function updateInternationalId(document: Omit<InternationalIdDocument, 'section' | 'type'>) {
  const apollo = getOrCreateClient({});

  const images: DocumentImageInput[] = document.images.map((image) => ({
    id: image.id,
    originalHeight: image.originalHeight,
    originalUri: image.originalUri,
    originalUrl: image.originalUrl,
    originalWidth: image.originalWidth,
    previewHeight: image.previewHeight,
    previewUri: image.previewUri,
    previewUrl: image.previewUrl,
    previewWidth: image.previewWidth,
  }));

  const result = await apollo.mutate<
    UpdateInternationalIdMutation,
    UpdateInternationalIdMutationVariables
  >({
    mutation: UpdateInternationalIdDocument,
    variables: {
      input: {
        type: DocumentType.InternationalId,
        internationalIdInput: {
          id: document.id,
          verificationStatus: document.verificationStatus,
          images,
          docNumber: document.docNumber,
          lastName: document.lastName,
          lastNameLat: document.lastNameLat,
          firstName: document.firstName,
          firstNameLat: document.firstNameLat,
          citizenship: document.citizenship,
          citizenshipLat: document.citizenshipLat,
          birthDate: document.birthDate,
          gender: document.gender,
          birthPlace: document.birthPlace,
          birthPlaceLat: document.birthPlaceLat,
          endDate: document.endDate,
          issueDate: document.issueDate,
          issuedBy: document.issuedBy,
        },
      },
      id: document.id,
    },
  });

  if (!result.data || result.data.updateDocument.__typename === 'UpdateDocumentProblem') {
    return {
      status: 'error',
      reason: UpdateDocumentProblemReason.Internal,
    };
  }

  return {
    status: 'ok',
    data:
      result.data.updateDocument.__typename === 'UpdateDocumentPayload' &&
      result.data.updateDocument.query.document,
  };
}

async function uploadImage<T>(file: File, id?: string, save = true) {
  const form = new FormData();

  form.append('file', file);
  form.append('type', DocumentType.InternationalId);
  form.append('save', save ? 'true' : 'false');

  if (id) {
    form.append('id', id);
  }

  const response = await request<T>({
    url: `/documents/upload`,
    method: 'POST',
    data: form,
  });

  return response;
}

async function uploadImages<T>(files: FileList, id: string, save = true) {
  const uploadRequests = Object.values(files).reduce<Promise<AxiosResponse<T, any>>[]>(
    (acc, file) => {
      if (file.size < MAX_FILE_SIZE_BYTES) {
        const uploadDocumentData = uploadImage<T>(file, id, save);

        acc.push(uploadDocumentData);
      }

      return acc;
    },
    [],
  );

  return await Promise.all(uploadRequests);
}

async function shareImages(images: DocumentImage[]) {
  const files: File[] = [];

  for (let i = 0; i < images.length; i++) {
    const { originalUrl } = images[i];
    const path = new URL(originalUrl).pathname;
    const response = await request({
      url: 'avatars-proxy',
      params: { path },
      responseType: 'arraybuffer',
    });

    const type = response.headers['content-type'];
    const [_, extension] = type.split('/');

    files.push(new File([new Uint8Array(response.data)], `image${i + 1}.${extension}`, { type }));
  }

  // Надо оборачивать в трайкетч, иначе отмена шаринга бросает исключение
  try {
    await navigator.share({
      files,
    });
  } catch (e) {}
}
