import { attach, createEvent, createStore, sample } from 'effector';
import { createGate } from 'effector-react';

import { DocumentGender, DocumentImage, InternationalIdDocument } from '@client/shared/api/graphql';
import { createForm, rules } from '@client/shared/libs/effector-forms';
import { createDialogApi } from '@client/shared/libs/effector-helpers/dialog';

import { MAX_FILE_SIZE_BYTES } from '../../constants';
import { DocumentNotifier } from '../../notitier';
import * as api from './api';
import { InternationalIdFormFields } from './interface';
import * as viewerModel from './viewer-model';

interface FormId {
  id?: string;
}

export const FormIdGate = createGate<FormId>();

// храним картинки в отдельном сторе, они могут удаляться, добавляться
// при сохранении документа отправляем картинки с этого стора
export const $images = createStore<DocumentImage[]>([]);
// Modal
export const dialog = createDialogApi('international-id-form');
export const confirmationDialog = createDialogApi();
export const deleteImageDialog = createDialogApi();
export const imageViewerDialog = createDialogApi();

// Events
export const closeFormClicked = createEvent();
export const confirmCloseFormClicked = createEvent();
export const closeConfirmationDialogCliked = createEvent();

export const imageCliked = createEvent();
export const deleteImageClicked = createEvent();
export const confirmDeleteImageClicked = createEvent<{ imageId: string }>();
export const abortDeleteImageCliked = createEvent();

const resetImages = createEvent();

// Form
export const form = createForm<InternationalIdFormFields>({
  fields: {
    images: {
      init: null as FileList | null,
    },
    id: {
      init: '',
    },
    docNumber: {
      init: '',
      rules: [rules.regexp(/^($|\d{9})$/)],
      filter: (value) => /^($|\d+)$/.test(value),
    },
    lastName: {
      init: '',
    },
    lastNameLat: {
      init: '',
    },
    firstName: {
      init: '',
    },
    firstNameLat: {
      init: '',
    },
    citizenship: {
      init: '',
    },
    citizenshipLat: {
      init: '',
    },
    birthDate: {
      init: null as Date | null,
    },
    gender: {
      init: null as DocumentGender | null,
    },
    birthPlace: {
      init: '',
    },
    birthPlaceLat: {
      init: '',
    },
    issueDate: {
      init: null as Date | null,
    },
    endDate: {
      init: null as Date | null,
    },
    issuedBy: {
      init: '',
    },
  },
  validateOn: ['change', 'blur', 'submit'],
});

// saveInternationalIdFormDataFx
const saveInternationalIdFormDataFx = attach({
  effect: api.saveInternationalIdFormDataFx,
  // source: form.$values,
});

// getInternationalIdDataFx
const getInternationalIdDataFx = attach({
  effect: api.getInternationalIdDataFx,
});

// deleteImageFx
const deleteImageFx = attach({
  effect: api.temporaryDeleteImageFx,
});

// uploadImagesFx
const uploadImagesFx = attach({
  effect: api.uploadImagesFx,
});

export const $isLoading = getInternationalIdDataFx.pending;
export const $isPending = saveInternationalIdFormDataFx.pending;
export const $isImagesUploading = uploadImagesFx.pending;
export const $isImageDeleting = deleteImageFx.pending;

// Обновления стора картинок после удаления / добавления
$images.on(uploadImagesFx.doneData, (currImages, addedImages) => [...currImages, ...addedImages]);
$images.on(deleteImageFx.doneData, (_, newImages) => newImages);
// Очистка стора после выхода
$images.on(resetImages, () => []);

// Получение данных по id документа
sample({
  clock: FormIdGate.state,
  target: [form.reset, resetImages, getInternationalIdDataFx],
});

sample({
  clock: getInternationalIdDataFx.failData,
  target: [dialog.hide, DocumentNotifier.notFound],
});

// Сохраняем данные в форму
sample({
  clock: getInternationalIdDataFx.doneData,
  filter(document): document is InternationalIdDocument {
    return (
      document !== null && document.verificationStatus !== 'full' && dialog.$isVisible.getState()
    );
  },
  fn: (document: InternationalIdDocument) => {
    return {
      id: document.id,
      docNumber: document.docNumber ?? '',
      lastName: document.lastName ?? '',
      lastNameLat: document.lastNameLat ?? '',
      firstName: document.firstName ?? '',
      firstNameLat: document.firstNameLat ?? '',
      citizenship: document.citizenship ?? '',
      citizenshipLat: document.citizenshipLat ?? '',
      birthDate: document?.birthDate ? new Date(document.birthDate) : null,
      gender: document.gender,
      birthPlace: document.birthPlace ?? '',
      birthPlaceLat: document.birthPlaceLat ?? '',
      issueDate: document.issueDate ? new Date(document.issueDate) : null,
      endDate: document.endDate ? new Date(document.endDate) : null,
      issuedBy: document.issuedBy ?? '',
    };
  },
  target: form.setForm,
});

// verificationStatus === full
sample({
  clock: getInternationalIdDataFx.doneData,
  filter(document): document is InternationalIdDocument {
    return document?.verificationStatus === 'full';
  },
  target: [dialog.hide, DocumentNotifier.cannotEdit],
});

// Сохраняем начальные картинки в стор картинок
sample({
  clock: getInternationalIdDataFx.doneData,
  filter(document): document is InternationalIdDocument {
    return (
      document !== null && document.verificationStatus !== 'full' && dialog.$isVisible.getState()
    );
  },
  fn: (document: InternationalIdDocument) => document.images,
  target: $images,
});

// Отправка данных
sample({
  clock: form.formValidated,
  source: { formData: form.$values, images: $images },
  target: saveInternationalIdFormDataFx,
});

// если это новый документ (нет id),
// то показываем снек об успешном добавлении докумнета
sample({
  clock: saveInternationalIdFormDataFx.doneData,
  source: form.$values,
  filter: (form) => !form.id,
  target: [DocumentNotifier.documentAdded.prepend(() => 'Загранпаспорт'), dialog.hide],
});

// иначе закрываем форму и открываем карточку документа
sample({
  clock: saveInternationalIdFormDataFx.doneData,
  source: form.$values,
  filter: (form) => Boolean(form.id),
  target: dialog.hide,
});

sample({
  clock: dialog.hide,
  source: form.$values.map((formData) => ({ id: formData.id })),
  filter: (form) => Boolean(form.id),
  target: [viewerModel.dialog.setParams, viewerModel.dialog.show],
});

// Просмотр картинок
sample({
  clock: imageCliked,
  target: imageViewerDialog.show,
});

// Загрузка картинок
sample({
  clock: form.fields.images.changed,
  source: { images: form.fields.images.$value, formData: form.$values },
  target: uploadImagesFx,
});

// Обработка больших картинок
sample({
  clock: form.fields.images.changed,
  filter: (images) => {
    if (images) {
      return Object.values(images).some((image) => image.size > MAX_FILE_SIZE_BYTES);
    }

    return false;
  },
  target: DocumentNotifier.maxFileSize,
});

// Удаление картинки
sample({
  clock: deleteImageClicked,
  target: deleteImageDialog.show,
});

sample({
  clock: abortDeleteImageCliked,
  target: deleteImageDialog.hide,
});

sample({
  clock: confirmDeleteImageClicked,
  source: $images,
  fn: (images, image) => ({ images, imageId: image.imageId }),
  target: deleteImageFx,
});

sample({
  clock: deleteImageFx.doneData,
  target: [deleteImageDialog.hide, imageViewerDialog.hide],
});

// Захотели выйти и есть данные - спрашиваем
sample({
  source: closeFormClicked,
  filter: form.$isTouched,
  target: confirmationDialog.show,
});

// Захотели выйти и нет данных - выходим
sample({
  source: closeFormClicked,
  filter: form.$isTouched.map((isTouched) => !isTouched),
  target: [confirmationDialog.hide, dialog.hide],
});

// Подтвердили выход - скрываем всё
sample({
  clock: confirmCloseFormClicked,
  target: [confirmationDialog.hide, dialog.hide, resetImages],
});

// Передумали выходить - остаемся на форме
sample({
  clock: closeConfirmationDialogCliked,
  target: confirmationDialog.hide,
});

// Очищаем состояние после закрытия формы
sample({
  clock: dialog.hide,
  target: [form.reset, resetImages],
});
