import { attach, createEvent, sample } from 'effector';

import { InternationalIdDocument } from '@client/shared/api/graphql';
import { createForm } from '@client/shared/libs/effector-forms';
import { createDialogApi } from '@client/shared/libs/effector-helpers';

import { MAX_FILE_SIZE_BYTES } from '../../constants';
import { DocumentNotifier } from '../../notitier';
import * as api from './api';
import * as viewerModel from './viewer-model';

export const deleteImageDialog = createDialogApi();
export const imageViewerDialog = createDialogApi();

export const imageCliked = createEvent();
export const deleteImageClicked = createEvent();
export const confirmDeleteImageClicked = createEvent<{ imageId: string }>();
export const abortDeleteImageCliked = createEvent();

export const imagesForm = createForm({
  fields: {
    id: {
      init: '',
    },
    images: {
      init: null as FileList | null,
    },
  },
});

const deleteImageFx = attach({
  effect: api.deleteImageFx,
});

const uploadAndSaveImagesFx = attach({
  effect: api.uploadAndSaveImagesFx,
});

export const $isImageDeleting = deleteImageFx.pending;
export const $isImagesUploading = uploadAndSaveImagesFx.pending;

sample({
  clock: imageCliked,
  target: imageViewerDialog.show,
});

// delete image
sample({
  clock: deleteImageClicked,
  target: deleteImageDialog.show,
});

sample({
  clock: abortDeleteImageCliked,
  target: deleteImageDialog.hide,
});

sample({
  clock: confirmDeleteImageClicked,
  source: viewerModel.$document,
  filter(document): document is InternationalIdDocument {
    return document !== null;
  },
  fn: (document: InternationalIdDocument, image) => ({ document, imageId: image.imageId }),
  target: deleteImageFx,
});

sample({
  clock: deleteImageFx.doneData,
  target: viewerModel.$document,
});

sample({
  clock: deleteImageFx.doneData,
  target: [viewerModel.$document, deleteImageDialog.hide, imageViewerDialog.hide],
});

// upload image
sample({
  clock: imagesForm.fields.images.changed,
  source: imagesForm.$values.map((state) => state),
  filter(form): form is { id: string; images: FileList } {
    return form.images !== null;
  },
  fn: (form: { id: string; images: FileList }) => ({ ...form }),
  target: uploadAndSaveImagesFx,
});

// Обработка больших картинок
sample({
  clock: imagesForm.fields.images.changed,
  filter: (images) => {
    if (images) {
      return Object.values(images).some((image) => image.size > MAX_FILE_SIZE_BYTES);
    }

    return false;
  },
  target: DocumentNotifier.maxFileSize,
});

sample({
  clock: uploadAndSaveImagesFx.doneData,
  target: viewerModel.$document,
});
