import Axios, { AxiosResponse } from 'axios';
import { createEffect } from 'effector';

import {
  AddDocumentProblemReason,
  AddMedicalInsuranceDocument,
  AddMedicalInsuranceMutation,
  AddMedicalInsuranceMutationVariables,
  DeleteDocumentInput,
  DeleteDocumentProblemReason,
  DeleteMedicalInsuranceDocumentDocument,
  DeleteMedicalInsuranceDocumentMutation,
  DeleteMedicalInsuranceDocumentMutationVariables,
  DocumentByIdProblemReason,
  DocumentImage,
  DocumentImageInput,
  DocumentType,
  GetMedicalInsuranceByIdDocument,
  GetMedicalInsuranceByIdQuery,
  GetMedicalInsuranceByIdQueryVariables,
  MedicalInsuranceDocument,
  UpdateDocumentProblemReason,
  UpdateMedicalInsuranceDocument,
  UpdateMedicalInsuranceMutation,
  UpdateMedicalInsuranceMutationVariables,
} from '@client/shared/api/graphql';
import { getOrCreateClient } from '@client/shared/libs/apollo';

import { MAX_FILE_SIZE_BYTES } from '../../../constants';
import { formatDateToISO } from '../../../libs/date';
import { MedicalInsuranceFormFields } from '../interface';
import { Field } from '../types/field';

const { request } = Axios.create({
  baseURL: '/personal/api/',
  // TODO: вот тут вопрос
  // сколько ставить
  // а еще мб как-нибудь ограничить рамер файла?
  timeout: 10000,
});

export interface DocumentId {
  id?: string;
}

export interface SaveMedicalInsuranceFormDataFxInput {
  formData: MedicalInsuranceFormFields;
  images: DocumentImage[];
}

export interface UploadImagesFxInput {
  formData: MedicalInsuranceFormFields;
  images: FileList | null;
}

export interface UploadAndSaveImagesFxInput {
  id: string;
  images: FileList;
}

export interface DeleteImageFxInput {
  document: MedicalInsuranceDocument;
  imageId: string;
}

interface DeleteDocumentError {
  reason: DeleteDocumentProblemReason;
}

interface GetMedicalInsuranceError {
  reason: DocumentByIdProblemReason;
}

interface TemporaryDeleteImageFxInput {
  images: DocumentImage[];
  imageId: string;
}

export const saveMedicalInsuranceFormDataFx = createEffect(
  async (input: SaveMedicalInsuranceFormDataFxInput) => {
    const { formData, images } = input;

    if (formData.id) {
      const response = await updateMedicalInsurance({
        id: formData.id,
        images,
        docNumber: formData.docNumber,
        lastName: formData.lastName,
        firstName: formData.firstName,
        middleName: formData.middleName,
        birthDate: formatDateToISO(formData.birthDate),
        gender: formData.gender,
        blank: formData.blank,
      });

      if (response.status === 'error') {
        return Promise.reject({ reason: response.reason });
      }
    } else {
      const response = await addMedicalInsurance({
        images,
        docNumber: formData.docNumber,
        lastName: formData.lastName,
        firstName: formData.firstName,
        middleName: formData.middleName,
        birthDate: formatDateToISO(formData.birthDate),
        gender: formData.gender,
        blank: formData.blank,
      });

      if (response.status === 'error') {
        return Promise.reject({ reason: response.reason });
      }
    }
  },
);

export const deleteMedicalInsuranceFx = createEffect<
  DeleteDocumentInput,
  string,
  DeleteDocumentError
>(async (input) => {
  const apollo = getOrCreateClient({});

  const result = await apollo.mutate<
    DeleteMedicalInsuranceDocumentMutation,
    DeleteMedicalInsuranceDocumentMutationVariables
  >({
    mutation: DeleteMedicalInsuranceDocumentDocument,
    variables: {
      input,
    },
  });

  if (!result.data || result.data.deleteDocument.__typename === 'DeleteDocumentProblem') {
    return Promise.reject({
      reason: DeleteDocumentProblemReason.Internal,
    });
  }

  return 'ok';
});

export const getMedicalInsuranceDataFx = createEffect<
  DocumentId,
  MedicalInsuranceDocument | null,
  GetMedicalInsuranceError
>(async (input) => {
  if (input.id) {
    const apollo = getOrCreateClient({});

    const result = await apollo.query<
      GetMedicalInsuranceByIdQuery,
      GetMedicalInsuranceByIdQueryVariables
    >({
      query: GetMedicalInsuranceByIdDocument,
      variables: {
        id: input.id,
      },
    });

    if (
      result.data.document.__typename === 'MedicalInsuranceDocument' &&
      result.data.document.type === DocumentType.MedicalInsurance
    ) {
      return result.data.document;
    }

    return Promise.reject({
      reason: DocumentByIdProblemReason.NotFound,
    });
  }

  return null;
});

export const uploadImagesFx = createEffect(async (input: UploadImagesFxInput) => {
  const { images, formData } = input;

  if (images) {
    const responses = await uploadImages<DocumentImage>(images, formData.id ?? '', false);

    const uploadedImages = responses.map((response) => response.data);

    return uploadedImages;
  }

  return [];
});

export const uploadAndSaveImagesFx = createEffect<
  UploadAndSaveImagesFxInput,
  MedicalInsuranceDocument
>(async (input) => {
  const { images: files, id } = input;

  await uploadImages<Document>(files, id, true);

  const apollo = getOrCreateClient({});

  const result = await apollo.query<
    GetMedicalInsuranceByIdQuery,
    GetMedicalInsuranceByIdQueryVariables
  >({
    query: GetMedicalInsuranceByIdDocument,
    fetchPolicy: 'network-only',
    variables: {
      id,
    },
  });

  if (result.data.document.__typename !== 'MedicalInsuranceDocument') {
    return Promise.reject({});
  }

  return result.data.document;
});

export const deleteImageFx = createEffect<
  DeleteImageFxInput,
  MedicalInsuranceDocument,
  { response: any }
>(async (input) => {
  const { document, imageId } = input;

  const images = document.images.filter((image) => image.id !== imageId);

  const response = await updateMedicalInsurance({
    ...document,
    images,
  });

  if (
    response.status === 'error' ||
    !response.data ||
    response.data.__typename !== 'MedicalInsuranceDocument'
  ) {
    return Promise.reject({ reason: response.reason });
  }

  return response.data;
});

export const temporaryDeleteImageFx = createEffect((input: TemporaryDeleteImageFxInput) => {
  const { images, imageId } = input;
  const newImages = images.filter((image) => image.id !== imageId);

  return newImages;
});

export const shareDocumentTextFx = createEffect<Field[], void, void>(async (fields) => {
  // Надо оборачивать в трайкетч, иначе отмена шаринга бросает исключение
  try {
    await navigator.share({
      title: 'ОМС',
      text: fields.reduce((acc, value) => acc + `${value.label}: ${value.text}\n`, ''),
    });
  } catch (e) {}
});

export const shareImagesFx = createEffect(shareImages);

export const shareImageFx = createEffect<DocumentImage, void, void>(async (image) => {
  return shareImages([image]);
});

async function addMedicalInsurance(
  document: Omit<MedicalInsuranceDocument, 'id' | 'section' | 'type'>,
) {
  const apollo = getOrCreateClient({});

  const images: DocumentImageInput[] = document.images.map((image) => ({
    id: image.id,
    originalHeight: image.originalHeight,
    originalUri: image.originalUri,
    originalUrl: image.originalUrl,
    originalWidth: image.originalWidth,
    previewHeight: image.previewHeight,
    previewUri: image.previewUri,
    previewUrl: image.previewUrl,
    previewWidth: image.previewWidth,
  }));

  const result = await apollo.mutate<
    AddMedicalInsuranceMutation,
    AddMedicalInsuranceMutationVariables
  >({
    mutation: AddMedicalInsuranceDocument,
    variables: {
      input: {
        type: DocumentType.MedicalInsurance,
        medicalInsuranceInput: {
          images,
          docNumber: document.docNumber,
          firstName: document.firstName,
          lastName: document.lastName,
          middleName: document.middleName,
          birthDate: document.birthDate,
          gender: document.gender,
          blank: document.blank,
        },
      },
    },
  });

  if (!result.data || result.data.addDocument.__typename === 'AddDocumentProblem') {
    return {
      status: 'error',
      reason: AddDocumentProblemReason.Internal,
    };
  }

  return {
    status: 'ok',
  };
}

async function updateMedicalInsurance(
  document: Omit<MedicalInsuranceDocument, 'section' | 'type'>,
) {
  const apollo = getOrCreateClient({});

  const images: DocumentImageInput[] = document.images.map((image) => ({
    id: image.id,
    originalHeight: image.originalHeight,
    originalUri: image.originalUri,
    originalUrl: image.originalUrl,
    originalWidth: image.originalWidth,
    previewHeight: image.previewHeight,
    previewUri: image.previewUri,
    previewUrl: image.previewUrl,
    previewWidth: image.previewWidth,
  }));

  const result = await apollo.mutate<
    UpdateMedicalInsuranceMutation,
    UpdateMedicalInsuranceMutationVariables
  >({
    mutation: UpdateMedicalInsuranceDocument,
    variables: {
      input: {
        type: DocumentType.MedicalInsurance,
        medicalInsuranceInput: {
          id: document.id,
          verificationStatus: document.verificationStatus,
          images,
          docNumber: document.docNumber,
          firstName: document.firstName,
          lastName: document.lastName,
          middleName: document.middleName,
          birthDate: document.birthDate,
          gender: document.gender,
          blank: document.blank,
        },
      },
      id: document.id,
    },
  });

  if (!result.data || result.data.updateDocument.__typename === 'UpdateDocumentProblem') {
    return {
      status: 'error',
      reason: UpdateDocumentProblemReason.Internal,
    };
  }

  return {
    status: 'ok',
    data:
      result.data.updateDocument.__typename === 'UpdateDocumentPayload' &&
      result.data.updateDocument.query.document,
  };
}

async function uploadImage<T>(file: File, id?: string, save = true) {
  const form = new FormData();

  form.append('file', file);
  form.append('type', DocumentType.MedicalInsurance);
  form.append('save', save ? 'true' : 'false');

  if (id) {
    form.append('id', id);
  }

  const response = await request<T>({
    url: `/documents/upload`,
    method: 'POST',
    data: form,
  });

  return response;
}

async function uploadImages<T>(files: FileList, id: string, save = true) {
  const uploadRequests = Object.values(files).reduce<Promise<AxiosResponse<T, any>>[]>(
    (acc, file) => {
      if (file.size < MAX_FILE_SIZE_BYTES) {
        const uploadDocumentData = uploadImage<T>(file, id, save);

        acc.push(uploadDocumentData);
      }

      return acc;
    },
    [],
  );

  return await Promise.all(uploadRequests);
}

async function shareImages(images: DocumentImage[]) {
  const files: File[] = [];

  for (let i = 0; i < images.length; i++) {
    const { originalUrl } = images[i];
    const path = new URL(originalUrl).pathname;
    const response = await request({
      url: 'avatars-proxy',
      params: { path },
      responseType: 'arraybuffer',
    });

    const type = response.headers['content-type'];
    const [_, extension] = type.split('/');

    files.push(new File([new Uint8Array(response.data)], `image${i + 1}.${extension}`, { type }));
  }

  // Надо оборачивать в трайкетч, иначе отмена шаринга бросает исключение
  try {
    await navigator.share({
      files,
    });
  } catch (e) {}
}
