import { attach, createEvent, createStore, sample } from 'effector';
import { createGate } from 'effector-react';

import { DocumentGender, DocumentImage, NationalIdDocument } from '@client/shared/api/graphql';
import { createForm, rules } from '@client/shared/libs/effector-forms';
import { createDialogApi } from '@client/shared/libs/effector-helpers/dialog';

import { MAX_FILE_SIZE_BYTES } from '../../constants';
import { DocumentNotifier } from '../../notitier';
import * as api from './api';
import { NationalIdFormFields } from './interface';
import * as viewerModel from './viewer-model';

interface FormId {
  id?: string;
}

export const FormIdGate = createGate<FormId>();

// храним картинки в отдельном сторе, они могут удаляться, добавляться
// при сохранении документа отправляем картинки с этого стора
export const $images = createStore<DocumentImage[]>([]);
// Modal
export const dialog = createDialogApi('passport-form');
export const confirmationDialog = createDialogApi();
export const deleteImageDialog = createDialogApi();
export const imageViewerDialog = createDialogApi();

// Events
export const closeFormClicked = createEvent();
export const confirmCloseFormClicked = createEvent();
export const closeConfirmationDialogCliked = createEvent();

export const imageCliked = createEvent();
export const deleteImageClicked = createEvent();
export const confirmDeleteImageClicked = createEvent<{ imageId: string }>();
export const abortDeleteImageCliked = createEvent();

const resetImages = createEvent();

// Form
export const form = createForm<NationalIdFormFields>({
  fields: {
    images: {
      init: null as FileList | null,
    },
    id: {
      init: '',
    },
    docNumber: {
      init: '',
      rules: [rules.regexp(/^($|\d{10})$/)],
      filter: (value) => /^($|\d+)$/.test(value),
    },
    lastName: {
      init: '',
    },
    firstName: {
      init: '',
    },
    middleName: {
      init: '',
    },
    birthDate: {
      init: null as Date | null,
    },
    gender: {
      init: null as DocumentGender | null,
    },
    birthPlace: {
      init: '',
    },
    issuedBy: {
      init: '',
    },
    issuerSubdivisionCode: {
      init: '',
      rules: [rules.regexp(/^($|\d{6})$/)],
      filter: (value) => /^($|\d+)$/.test(value),
      validateOn: ['blur'],
    },
    issueDate: {
      init: null as Date | null,
    },
    registrationAddress: {
      init: '',
    },
  },
  validateOn: ['change', 'blur', 'submit'],
});

// saveNationalIdFormDataFx
const saveNationalIdFormDataFx = attach({
  effect: api.saveNationalIdFormDataFx,
});

// getNationalIdDataFx
const getNationalIdDataFx = attach({
  effect: api.getNationalIdDataFx,
});

// deleteImageFx
const deleteImageFx = attach({
  effect: api.temporaryDeleteImageFx,
});

// uploadImagesFx
const uploadImagesFx = attach({
  effect: api.uploadImagesFx,
});

export const $isLoading = getNationalIdDataFx.pending;
export const $isPending = saveNationalIdFormDataFx.pending;
export const $isImagesUploading = uploadImagesFx.pending;
export const $isImageDeleting = deleteImageFx.pending;

// Обновления стора картинок после удаления / добавления
$images.on(uploadImagesFx.doneData, (currImages, addedImages) => [...currImages, ...addedImages]);
$images.on(deleteImageFx.doneData, (_, newImages) => newImages);
// Очистка стора после выхода
$images.on(resetImages, () => []);

// Получение данных по id документа
sample({
  clock: FormIdGate.state,
  target: [form.reset, resetImages, getNationalIdDataFx],
});

sample({
  clock: getNationalIdDataFx.failData,
  target: [dialog.hide, DocumentNotifier.notFound],
});

// Сохраняем данные в форму
sample({
  clock: getNationalIdDataFx.doneData,
  filter(document): document is NationalIdDocument {
    return (
      document !== null && document.verificationStatus !== 'full' && dialog.$isVisible.getState()
    );
  },
  fn: (document: NationalIdDocument) => {
    return {
      id: document.id,
      docNumber: document.docNumber ?? '',
      lastName: document.lastName ?? '',
      firstName: document.firstName ?? '',
      middleName: document.middleName ?? '',
      birthDate: document?.birthDate ? new Date(document.birthDate) : null,
      gender: document.gender,
      birthPlace: document.birthPlace ?? '',
      issueDate: document.issueDate ? new Date(document.issueDate) : null,
      issuedBy: document.issuedBy ?? '',
      issuerSubdivisionCode: document.issuerSubdivisionCode ?? '',
      registrationAddress: document.registrationAddress ?? '',
    };
  },
  target: form.setForm,
});

// Не открываем форму, если документ нельзя редактировать
// verificationStatus === full
sample({
  clock: getNationalIdDataFx.doneData,
  filter(document): document is NationalIdDocument {
    return document?.verificationStatus === 'full';
  },
  target: [dialog.hide, DocumentNotifier.cannotEdit],
});

// Сохраняем начальные картинки в стор картинок
sample({
  clock: getNationalIdDataFx.doneData,
  filter(document): document is NationalIdDocument {
    return (
      document !== null && document.verificationStatus !== 'full' && dialog.$isVisible.getState()
    );
  },
  fn: (document: NationalIdDocument) => document.images,
  target: $images,
});

// Отправка данных
sample({
  clock: form.formValidated,
  source: { formData: form.$values, images: $images },
  target: saveNationalIdFormDataFx,
});

// если это новый документ (нет id),
// то показываем снек об успешном добавлении докумнета
sample({
  clock: saveNationalIdFormDataFx.doneData,
  source: form.$values,
  filter: (form) => !form.id,
  target: [DocumentNotifier.documentAdded.prepend(() => 'Паспорт РФ'), dialog.hide],
});

// иначе закрываем форму и открываем карточку документа
sample({
  clock: saveNationalIdFormDataFx.doneData,
  source: form.$values,
  filter: (form) => Boolean(form.id),
  target: dialog.hide,
});

sample({
  clock: dialog.hide,
  source: form.$values.map((formData) => ({ id: formData.id })),
  filter: (form) => Boolean(form.id),
  target: [viewerModel.dialog.setParams, viewerModel.dialog.show],
});

// Просмотр картинок
sample({
  clock: imageCliked,
  target: imageViewerDialog.show,
});

// Загрузка картинок
sample({
  clock: form.fields.images.changed,
  source: { images: form.fields.images.$value, formData: form.$values },
  target: uploadImagesFx,
});

// Обработка больших картинок
sample({
  clock: form.fields.images.changed,
  filter: (images) => {
    if (images) {
      return Object.values(images).some((image) => image.size > MAX_FILE_SIZE_BYTES);
    }

    return false;
  },
  target: DocumentNotifier.maxFileSize,
});

// Удаление картинки
sample({
  clock: deleteImageClicked,
  target: deleteImageDialog.show,
});

sample({
  clock: abortDeleteImageCliked,
  target: deleteImageDialog.hide,
});

sample({
  clock: confirmDeleteImageClicked,
  source: $images,
  fn: (images, image) => ({ images, imageId: image.imageId }),
  target: deleteImageFx,
});

sample({
  clock: deleteImageFx.doneData,
  target: [deleteImageDialog.hide, imageViewerDialog.hide],
});

// Захотели выйти и есть данные - спрашиваем
sample({
  clock: closeFormClicked,
  filter: form.$isTouched,
  target: confirmationDialog.show,
});

// Захотели выйти и нет данных - выходим
sample({
  clock: closeFormClicked,
  filter: form.$isTouched.map((isTouched) => !isTouched),
  target: [confirmationDialog.hide, dialog.hide],
});

// Подтвердили выход - скрываем всё, очищаем состояние
sample({
  clock: confirmCloseFormClicked,
  target: [confirmationDialog.hide, dialog.hide, resetImages],
});

// Передумали выходить - остаемся на форме
sample({
  clock: closeConfirmationDialogCliked,
  target: confirmationDialog.hide,
});

// Очищаем состояние после закрытия формы
sample({
  clock: dialog.hide,
  target: [form.reset, resetImages],
});
