import { attach, createEvent, createStore, sample } from 'effector';
import { createGate } from 'effector-react';

import { DocumentGender, DocumentImage, SnilsDocument } from '@client/shared/api/graphql';
import { createForm, rules } from '@client/shared/libs/effector-forms';
import { createDialogApi } from '@client/shared/libs/effector-helpers/dialog';

import { MAX_FILE_SIZE_BYTES } from '../../constants';
import { DocumentNotifier } from '../../notitier';
import * as api from './api';
import { SnilsFormFields } from './interface';
import * as viewerModel from './viewer-model';

interface FormId {
  id?: string;
}

export const FormIdGate = createGate<FormId>();

// храним картинки в отдельном сторе, они могут удаляться, добавляться
// при сохранении документа отправляем картинки с этого стора
export const $images = createStore<DocumentImage[]>([]);
// Modal
export const dialog = createDialogApi('snils-form');
export const confirmationDialog = createDialogApi();
export const deleteImageDialog = createDialogApi();
export const imageViewerDialog = createDialogApi();

// Events
export const closeFormClicked = createEvent();
export const confirmCloseFormClicked = createEvent();
export const closeConfirmationDialogCliked = createEvent();

export const imageCliked = createEvent();
export const deleteImageClicked = createEvent();
export const confirmDeleteImageClicked = createEvent<{ imageId: string }>();
export const abortDeleteImageCliked = createEvent();

const resetImages = createEvent();

// Form
export const form = createForm<SnilsFormFields>({
  fields: {
    images: {
      init: null as FileList | null,
    },
    id: {
      init: '',
    },
    docNumber: {
      init: '',
      rules: [rules.requiredLength(11)],
      filter: (value) => /^($|\d+)$/.test(value),
    },
    lastName: {
      init: '',
    },
    firstName: {
      init: '',
    },
    middleName: {
      init: '',
    },
    birthDate: {
      init: null as Date | null,
    },
    birthPlace: {
      init: '',
    },
    gender: {
      init: null as DocumentGender | null,
    },
    issueDate: {
      init: null as Date | null,
    },
  },
  validateOn: ['change', 'blur', 'submit'],
});

// saveSnilsFormDataFx
const saveSnilsFormDataFx = attach({
  effect: api.saveSnilsFormDataFx,
});

// getSnilsDataFx
const getSnilsDataFx = attach({
  effect: api.getSnilsDataFx,
});

// deleteImageFx
const deleteImageFx = attach({
  effect: api.temporaryDeleteImageFx,
});

// uploadImagesFx
const uploadImagesFx = attach({
  effect: api.uploadImagesFx,
});

export const $isLoading = getSnilsDataFx.pending;
export const $isPending = saveSnilsFormDataFx.pending;
export const $isImagesUploading = uploadImagesFx.pending;
export const $isImageDeleting = deleteImageFx.pending;

// Обновления стора картинок после удаления / добавления
$images.on(uploadImagesFx.doneData, (currImages, addedImages) => [...currImages, ...addedImages]);
$images.on(deleteImageFx.doneData, (_, newImages) => newImages);
// Очистка стора
$images.reset([resetImages]);

// Получение данных по id документа
sample({
  clock: FormIdGate.state,
  target: [form.reset, resetImages, getSnilsDataFx],
});

sample({
  clock: getSnilsDataFx.failData,
  target: [dialog.hide, DocumentNotifier.notFound],
});

// Сохраняем данные в форму
sample({
  clock: getSnilsDataFx.doneData,
  filter(document): document is SnilsDocument {
    return (
      document !== null && document.verificationStatus !== 'full' && dialog.$isVisible.getState()
    );
  },
  fn: (document: SnilsDocument) => {
    return {
      id: document.id,
      docNumber: document.docNumber ?? '',
      lastName: document.lastName ?? '',
      firstName: document.firstName ?? '',
      middleName: document.middleName ?? '',
      birthDate: document?.birthDate ? new Date(document.birthDate) : null,
      birthPlace: document.birthPlace ?? '',
      gender: document.gender,
      issueDate: document.issueDate ? new Date(document.issueDate) : null,
    };
  },
  target: form.setForm,
});

// Не открываем форму, если документ нельзя редактировать
// verificationStatus === full
sample({
  clock: getSnilsDataFx.doneData,
  filter(document): document is SnilsDocument {
    return document?.verificationStatus === 'full';
  },
  target: [dialog.hide, DocumentNotifier.cannotEdit],
});

// Сохраняем начальные картинки в стор картинок
sample({
  clock: getSnilsDataFx.doneData,
  filter(document): document is SnilsDocument {
    return (
      document !== null && document.verificationStatus !== 'full' && dialog.$isVisible.getState()
    );
  },
  fn: (document: SnilsDocument) => document.images,
  target: $images,
});

// Отправка данных
sample({
  clock: form.formValidated,
  source: { formData: form.$values, images: $images },
  target: saveSnilsFormDataFx,
});

// если это новый документ (нет id),
// то показываем снек об успешном добавлении докумнета
sample({
  clock: saveSnilsFormDataFx.doneData,
  source: form.$values,
  filter: (form) => !form.id,
  target: [DocumentNotifier.documentAdded.prepend(() => 'СНИЛС'), dialog.hide],
});

// иначе закрываем форму и открываем карточку документа
sample({
  clock: saveSnilsFormDataFx.doneData,
  source: form.$values,
  filter: (form) => Boolean(form.id),
  target: dialog.hide,
});

sample({
  clock: dialog.hide,
  source: form.$values.map((formData) => ({ id: formData.id })),
  filter: (form) => Boolean(form.id),
  target: [viewerModel.dialog.setParams, viewerModel.dialog.show],
});

// Просмотр картинок
sample({
  clock: imageCliked,
  target: imageViewerDialog.show,
});

// Загрузка картинок
sample({
  clock: form.fields.images.changed,
  source: { images: form.fields.images.$value, formData: form.$values },
  target: uploadImagesFx,
});

// Обработка больших картинок
sample({
  clock: form.fields.images.changed,
  filter: (images) => {
    if (images) {
      return Object.values(images).some((image) => image.size > MAX_FILE_SIZE_BYTES);
    }

    return false;
  },
  target: DocumentNotifier.maxFileSize,
});

// Удаление картинки
sample({
  clock: deleteImageClicked,
  target: deleteImageDialog.show,
});

sample({
  clock: abortDeleteImageCliked,
  target: deleteImageDialog.hide,
});

sample({
  clock: confirmDeleteImageClicked,
  source: $images,
  fn: (images, image) => ({ images, imageId: image.imageId }),
  target: deleteImageFx,
});

sample({
  clock: deleteImageFx.doneData,
  target: [deleteImageDialog.hide, imageViewerDialog.hide],
});

// Захотели выйти и есть данные - спрашиваем
sample({
  source: closeFormClicked,
  filter: form.$touched,
  target: confirmationDialog.show,
});

// Захотели выйти и нет данных - выходим
sample({
  source: closeFormClicked,
  filter: form.$touched.map((touched) => !touched),
  target: [confirmationDialog.hide, dialog.hide],
});

// Подтвердили выход - скрываем всё
sample({
  clock: confirmCloseFormClicked,
  target: [confirmationDialog.hide, dialog.hide, resetImages],
});

// Передумали выходить - остаемся на форма
sample({
  clock: closeConfirmationDialogCliked,
  target: confirmationDialog.hide,
});

// Очищаем состояние после закрытия формы
sample({
  clock: dialog.hide,
  target: [form.reset, resetImages],
});
