import { useEvent, useGate, useStore } from 'effector-react';
import { VFC, useCallback, useState } from 'react';

import { CloseFormDialog, DocumentForm } from '@client/entities/document-form';
import { DeleteImageDialog, DocumentImageList } from '@client/entities/document-image-list';
import { useForm } from '@client/shared/libs/effector-forms';
import { useQueryParams } from '@client/shared/libs/router';
import { DateField, TextField } from '@client/shared/ui-kit';

import { VehicleRegistrationCertificateFormSkeleton } from './VehicleRegistrationCertificateFormSkeleton';
import * as model from './form-model';
import { FieldType } from './interface';

export interface VehicleRegistrationCertificateFormQueryParams {
  field?: FieldType;
  id?: string;
}

export const VehicleRegistrationCertificateForm: VFC = () => {
  const form = useForm(model.form);
  const queryParams = useQueryParams<VehicleRegistrationCertificateFormQueryParams>();

  const images = useStore(model.$images);

  const isVisible = useStore(model.dialog.$isVisible);
  const isLoading = useStore(model.$isLoading);
  const isPending = useStore(model.$isPending);
  const isImagesUploading = useStore(model.$isImagesUploading);
  const isImageDeleting = useStore(model.$isImageDeleting);
  const isImageViewerVisible = useStore(model.imageViewerDialog.$isVisible);
  const isConfirmationDialogVisible = useStore(model.confirmationDialog.$isVisible);
  const isDeleteImageDialogVisible = useStore(model.deleteImageDialog.$isVisible);

  const handleOnSubmit = useEvent(form.submit);
  const handleOnClose = useEvent(model.closeFormClicked);
  const handleOnCloseConfirm = useEvent(model.confirmCloseFormClicked);
  const handleOnCloseConfirmation = useEvent(model.closeConfirmationDialogCliked);

  const handleOnImageClicked = useEvent(model.imageCliked);
  const handleOnImageViewerClose = useEvent(model.imageViewerDialog.hide);
  const handleOnDeleteImage = useEvent(model.deleteImageClicked);
  const handleOnDeleteImageConfirm = useEvent(model.confirmDeleteImageClicked);
  const handleOnDeleteImageAbort = useEvent(model.abortDeleteImageCliked);

  useGate(model.FormIdGate, {
    id: isVisible ? queryParams.id ?? '' : '',
  });

  const handleOnImagesUpload = useCallback(
    (images: FileList | null) => {
      if (images && images.length > 0) {
        form.fields.images.onChange(images);
      }
    },
    [form.fields.images],
  );

  const [activeImageId, setActiveImageId] = useState<string | null>(null);

  const deleteImage = useCallback(() => {
    if (activeImageId) {
      handleOnDeleteImageConfirm({
        imageId: activeImageId,
      });
    }
  }, [activeImageId, handleOnDeleteImageConfirm]);

  if (!isVisible) {
    return null;
  }

  if (document === null || isLoading) {
    return <VehicleRegistrationCertificateFormSkeleton />;
  }

  return (
    <>
      <DocumentForm
        isVisible={isVisible}
        isDisabled={isPending || !(form.isDirty && form.isValid)}
        onClose={handleOnClose}
        onSubmit={handleOnSubmit}
      >
        <DocumentForm.Title text="СТС" />
        <DocumentForm.Preview>
          <DocumentImageList
            onImageClick={handleOnImageClicked}
            onImageViewerClose={handleOnImageViewerClose}
            isImageViewerVisible={isImageViewerVisible}
            onDelete={(id) => {
              setActiveImageId(id);
              handleOnDeleteImage();
            }}
            isUploading={isImagesUploading}
            onUpload={handleOnImagesUpload}
            isDownloadDisabled
            images={images}
            target="upload"
          />
        </DocumentForm.Preview>
        <DocumentForm.Section>
          <TextField
            autoFocus={queryParams.field === 'docNumber'}
            label="Серия и номер"
            inputMode="decimal"
            error={form.fields.docNumber.errorText({
              regexp: 'Введите серию и номер полностью (10 цифр)',
            })}
            value={form.fields.docNumber.value}
            onChangeValue={form.fields.docNumber.onChange}
          />
          <DocumentForm.MultiField>
            <TextField
              autoFocus={queryParams.field === 'lastName'}
              label="Фамилия"
              placeholder="Фамилия"
              value={form.fields.lastName.value}
              onChangeValue={form.fields.lastName.onChange}
            />
            <TextField
              placeholder="Фамилия латиницей"
              value={form.fields.lastNameLat.value}
              onChangeValue={form.fields.lastNameLat.onChange}
            />
          </DocumentForm.MultiField>
          <DocumentForm.MultiField>
            <TextField
              autoFocus={queryParams.field === 'firstName'}
              label="Имя"
              placeholder="Имя"
              value={form.fields.firstName.value}
              onChangeValue={form.fields.firstName.onChange}
            />
            <TextField
              placeholder="Имя латиницей"
              value={form.fields.firstNameLat.value}
              onChangeValue={form.fields.firstNameLat.onChange}
            />
          </DocumentForm.MultiField>
          <TextField
            autoFocus={queryParams.field === 'middleName'}
            label="Отчество"
            value={form.fields.middleName.value}
            onChangeValue={form.fields.middleName.onChange}
          />
          <TextField
            autoFocus={queryParams.field === 'registrationAddress'}
            label="Адрес регистрации"
            value={form.fields.registrationAddress.value}
            onChangeValue={form.fields.registrationAddress.onChange}
          />
          <TextField
            autoFocus={queryParams.field === 'issuerSubdivisionCode'}
            label="Код подразделения (выдано ГИБДД)"
            value={form.fields.issuerSubdivisionCode.value}
            onChangeValue={form.fields.issuerSubdivisionCode.onChange}
          />
          <DateField
            size="xl"
            label="Дата выдачи"
            autoFocus={queryParams.field === 'issueDate'}
            value={form.fields.issueDate.value}
            onChangeValue={form.fields.issueDate.onChange}
          />
          <TextField
            autoFocus={queryParams.field === 'additionalInformation'}
            label="Особые отметки"
            value={form.fields.additionalInformation.value}
            onChangeValue={form.fields.additionalInformation.onChange}
          />
        </DocumentForm.Section>
        <DocumentForm.Section>
          <TextField
            autoFocus={queryParams.field === 'carNumber'}
            label="Регистрационный знак"
            placeholder="X000XX00"
            value={form.fields.carNumber.value}
            onChangeValue={form.fields.carNumber.onChange}
          />
          <TextField
            autoFocus={queryParams.field === 'vin'}
            label="Идентификационный номер (VIN)"
            value={form.fields.vin.value}
            onChangeValue={form.fields.vin.onChange}
          />
          <DocumentForm.MultiField>
            <TextField
              autoFocus={queryParams.field === 'model'}
              label="Марка, модель"
              placeholder="Марка, модель"
              value={form.fields.model.value}
              onChangeValue={form.fields.model.onChange}
            />
            <TextField
              placeholder="Марка, модель латиницей"
              value={form.fields.modelLat.value}
              onChangeValue={form.fields.modelLat.onChange}
            />
          </DocumentForm.MultiField>
          <TextField
            autoFocus={queryParams.field === 'vehicleType'}
            label="Тип ТС"
            value={form.fields.vehicleType.value}
            onChangeValue={form.fields.vehicleType.onChange}
          />
          <TextField
            autoFocus={queryParams.field === 'category'}
            label="Категория ТС"
            value={form.fields.category.value}
            onChangeValue={form.fields.category.onChange}
          />
          <TextField
            autoFocus={queryParams.field === 'year'}
            label="Год выпуска"
            value={form.fields.year.value}
            onChangeValue={form.fields.year.onChange}
          />
          <TextField
            autoFocus={queryParams.field === 'shasiNumber'}
            label="Номер шасси (рамы)"
            value={form.fields.shasiNumber.value}
            onChangeValue={form.fields.shasiNumber.onChange}
          />
          <TextField
            autoFocus={queryParams.field === 'bodyNumber'}
            label="Номер кузова (кабины, прицепа)"
            value={form.fields.bodyNumber.value}
            onChangeValue={form.fields.bodyNumber.onChange}
          />
          <TextField
            autoFocus={queryParams.field === 'color'}
            label="Цвет"
            value={form.fields.color.value}
            onChangeValue={form.fields.color.onChange}
          />
          <TextField
            autoFocus={queryParams.field === 'power'}
            label="Мощность двигателя, кВт/л.с."
            value={form.fields.power.value}
            onChangeValue={form.fields.power.onChange}
          />
          <TextField
            autoFocus={queryParams.field === 'ecoClass'}
            label="Экологический класс"
            value={form.fields.ecoClass.value}
            onChangeValue={form.fields.ecoClass.onChange}
          />
          <TextField
            autoFocus={queryParams.field === 'maxWeight'}
            label="Разрешенная максимальная масса, кг"
            inputMode="decimal"
            value={form.fields.maxWeight.value}
            onChangeValue={form.fields.maxWeight.onChange}
          />
          <TextField
            autoFocus={queryParams.field === 'withoutLoadWeight'}
            label="Масса в снаряженном состоянии (без нагрузки), кг"
            inputMode="decimal"
            value={form.fields.withoutLoadWeight.value}
            onChangeValue={form.fields.withoutLoadWeight.onChange}
          />
          <TextField
            autoFocus={queryParams.field === 'temporaryReg'}
            label="Срок временной регистрации"
            value={form.fields.temporaryReg.value}
            onChangeValue={form.fields.temporaryReg.onChange}
          />
          <TextField
            autoFocus={queryParams.field === 'pts'}
            label="ПТС"
            value={form.fields.pts.value}
            onChangeValue={form.fields.pts.onChange}
          />
        </DocumentForm.Section>
      </DocumentForm>
      <CloseFormDialog
        isVisible={isConfirmationDialogVisible}
        onCloseDialog={handleOnCloseConfirmation}
        onContinueEditing={handleOnCloseConfirmation}
        onCloseForm={handleOnCloseConfirm}
      />
      <DeleteImageDialog
        isVisible={isDeleteImageDialogVisible}
        disabled={isImageDeleting}
        onConfirm={deleteImage}
        onAbort={handleOnDeleteImageAbort}
      />
    </>
  );
};
