import Axios, { AxiosResponse } from 'axios';
import { createEffect } from 'effector';

import {
  AddDocumentProblemReason,
  AddVehicleRegistrationCertificateDocument,
  AddVehicleRegistrationCertificateMutation,
  AddVehicleRegistrationCertificateMutationVariables,
  DeleteDocumentInput,
  DeleteDocumentProblemReason,
  DeleteVehicleRegistrationCertificateDocumentDocument,
  DeleteVehicleRegistrationCertificateDocumentMutation,
  DeleteVehicleRegistrationCertificateDocumentMutationVariables,
  DocumentByIdProblemReason,
  DocumentImage,
  DocumentImageInput,
  DocumentType,
  GetVehicleRegistrationCertificateByIdDocument,
  GetVehicleRegistrationCertificateByIdQuery,
  GetVehicleRegistrationCertificateByIdQueryVariables,
  UpdateDocumentProblemReason,
  UpdateVehicleRegistrationCertificateDocument,
  UpdateVehicleRegistrationCertificateMutation,
  UpdateVehicleRegistrationCertificateMutationVariables,
  VehicleRegistrationCertificateDocument,
} from '@client/shared/api/graphql';
import { getOrCreateClient } from '@client/shared/libs/apollo';

import { MAX_FILE_SIZE_BYTES } from '../../../constants';
import { formatDateToISO } from '../../../libs/date';
import { VehicleRegistrationCertificateFormFields } from '../interface';
import { Field } from '../types/field';

const { request } = Axios.create({
  baseURL: '/personal/api/',
  // TODO: вот тут вопрос
  // сколько ставить
  // а еще мб как-нибудь ограничить рамер файла?
  timeout: 10000,
});

export interface DocumentId {
  id?: string;
}

export interface SaveVehicleRegistrationCertificateFormDataFxInput {
  formData: VehicleRegistrationCertificateFormFields;
  images: DocumentImage[];
}

export interface UploadImagesFxInput {
  formData: VehicleRegistrationCertificateFormFields;
  images: FileList | null;
}

export interface UploadAndSaveImagesFxInput {
  id: string;
  images: FileList;
}

export interface DeleteImageFxInput {
  document: VehicleRegistrationCertificateDocument;
  imageId: string;
}

interface DeleteDocumentError {
  reason: DeleteDocumentProblemReason;
}

interface GetVehicleRegistrationCertificateError {
  reason: DocumentByIdProblemReason;
}

interface TemporaryDeleteImageFxInput {
  images: DocumentImage[];
  imageId: string;
}

export const saveVehicleRegistrationCertificateFormDataFx = createEffect(
  async (input: SaveVehicleRegistrationCertificateFormDataFxInput) => {
    const { formData, images } = input;

    if (formData.id) {
      const response = await updateVehicleRegistrationCertificate({
        id: formData.id,
        ...formData,
        images,
        issueDate: formatDateToISO(formData.issueDate),
        // TODO: сделать так же везде
        // docNumber: formData.docNumber,
        // lastName: formData.lastName,
        // lastNameLat: formData.lastNameLat,
        // firstName: formData.firstName,
        // firstNameLat: formData.firstNameLat,
        // middleName: formData.middleName,
        // registrationAddress: formData.registrationAddress,
        // issuerSubdivisionCode: formData.issuerSubdivisionCode,
        // additionalInformation: formData.additionalInformation,
        // carNumber: formData.carNumber,
        // vin: formData.vin,
        // model: formData.model,
        // modelLat: formData.modelLat,
        // vehicleType: formData.vehicleType,
        // category: formData.category,
        // year: formData.year,
        // shasiNumber: formData.shasiNumber,
        // bodyNumber: formData.bodyNumber,
        // color: formData.color,
        // power: formData.power,
        // ecoClass: formData.ecoClass,
        // maxWeight: formData.maxWeight,
        // withoutLoadWeight: formData.withoutLoadWeight,
        // temporaryReg: formData.temporaryReg,
        // pts: formData.pts,
      });

      if (response.status === 'error') {
        return Promise.reject({ reason: response.reason });
      }
    } else {
      const response = await addVehicleRegistrationCertificate({
        images,
        issueDate: formatDateToISO(formData.issueDate),
        docNumber: formData.docNumber,
        lastName: formData.lastName,
        lastNameLat: formData.lastNameLat,
        firstName: formData.firstName,
        firstNameLat: formData.firstNameLat,
        middleName: formData.middleName,
        registrationAddress: formData.registrationAddress,
        issuerSubdivisionCode: formData.issuerSubdivisionCode,
        additionalInformation: formData.additionalInformation,
        carNumber: formData.carNumber,
        vin: formData.vin,
        model: formData.model,
        modelLat: formData.modelLat,
        vehicleType: formData.vehicleType,
        category: formData.category,
        year: formData.year,
        shasiNumber: formData.shasiNumber,
        bodyNumber: formData.bodyNumber,
        color: formData.color,
        power: formData.power,
        ecoClass: formData.ecoClass,
        maxWeight: formData.maxWeight,
        withoutLoadWeight: formData.withoutLoadWeight,
        temporaryReg: formData.temporaryReg,
        pts: formData.pts,
      });

      if (response.status === 'error') {
        return Promise.reject({ reason: response.reason });
      }
    }
  },
);

export const deleteVehicleRegistrationCertificateFx = createEffect<
  DeleteDocumentInput,
  string,
  DeleteDocumentError
>(async (input) => {
  const apollo = getOrCreateClient({});

  const result = await apollo.mutate<
    DeleteVehicleRegistrationCertificateDocumentMutation,
    DeleteVehicleRegistrationCertificateDocumentMutationVariables
  >({
    mutation: DeleteVehicleRegistrationCertificateDocumentDocument,
    variables: {
      input,
    },
  });

  if (!result.data || result.data.deleteDocument.__typename === 'DeleteDocumentProblem') {
    return Promise.reject({
      reason: DeleteDocumentProblemReason.Internal,
    });
  }

  return 'ok';
});

export const getVehicleRegistrationCertificateDataFx = createEffect<
  DocumentId,
  VehicleRegistrationCertificateDocument | null,
  GetVehicleRegistrationCertificateError
>(async (input) => {
  if (input.id) {
    const apollo = getOrCreateClient({});

    const result = await apollo.query<
      GetVehicleRegistrationCertificateByIdQuery,
      GetVehicleRegistrationCertificateByIdQueryVariables
    >({
      query: GetVehicleRegistrationCertificateByIdDocument,
      variables: {
        id: input.id,
      },
    });

    if (
      result.data.document.__typename === 'VehicleRegistrationCertificateDocument' &&
      result.data.document.type === DocumentType.VehicleRegistrationCertificate
    ) {
      return result.data.document;
    }

    return Promise.reject({
      reason: DocumentByIdProblemReason.NotFound,
    });
  }

  return null;
});

export const uploadImagesFx = createEffect(async (input: UploadImagesFxInput) => {
  const { images, formData } = input;

  if (images) {
    const responses = await uploadImages<DocumentImage>(images, formData.id ?? '', false);

    const uploadedImages = responses.map((response) => response.data);

    return uploadedImages;
  }

  return [];
});

export const uploadAndSaveImagesFx = createEffect<
  UploadAndSaveImagesFxInput,
  VehicleRegistrationCertificateDocument
>(async (input) => {
  const { images: files, id } = input;

  await uploadImages<Document>(files, id, true);

  const apollo = getOrCreateClient({});

  const result = await apollo.query<
    GetVehicleRegistrationCertificateByIdQuery,
    GetVehicleRegistrationCertificateByIdQueryVariables
  >({
    query: GetVehicleRegistrationCertificateByIdDocument,
    fetchPolicy: 'network-only',
    variables: {
      id,
    },
  });

  if (result.data.document.__typename !== 'VehicleRegistrationCertificateDocument') {
    return Promise.reject({});
  }

  return result.data.document;
});

export const deleteImageFx = createEffect<
  DeleteImageFxInput,
  VehicleRegistrationCertificateDocument,
  { response: any }
>(async (input) => {
  const { document, imageId } = input;

  const images = document.images.filter((image) => image.id !== imageId);

  const response = await updateVehicleRegistrationCertificate({
    ...document,
    images,
  });

  if (
    response.status === 'error' ||
    !response.data ||
    response.data.__typename !== 'VehicleRegistrationCertificateDocument'
  ) {
    return Promise.reject({ reason: response.reason });
  }

  return response.data;
});

export const temporaryDeleteImageFx = createEffect((input: TemporaryDeleteImageFxInput) => {
  const { images, imageId } = input;
  const newImages = images.filter((image) => image.id !== imageId);

  return newImages;
});

export const shareDocumentTextFx = createEffect<Field[], void, void>(async (fields) => {
  // Надо оборачивать в трайкетч, иначе отмена шаринга бросает исключение
  try {
    await navigator.share({
      title: 'СТС',
      text: fields.reduce((acc, value) => acc + `${value.label}: ${value.text}\n`, ''),
    });
  } catch (e) {}
});

export const shareImagesFx = createEffect(shareImages);

export const shareImageFx = createEffect<DocumentImage, void, void>(async (image) => {
  return shareImages([image]);
});

async function addVehicleRegistrationCertificate(
  document: Omit<VehicleRegistrationCertificateDocument, 'id' | 'section' | 'type'>,
) {
  const apollo = getOrCreateClient({});

  const images: DocumentImageInput[] = document.images.map((image) => ({
    id: image.id,
    originalHeight: image.originalHeight,
    originalUri: image.originalUri,
    originalUrl: image.originalUrl,
    originalWidth: image.originalWidth,
    previewHeight: image.previewHeight,
    previewUri: image.previewUri,
    previewUrl: image.previewUrl,
    previewWidth: image.previewWidth,
  }));

  const result = await apollo.mutate<
    AddVehicleRegistrationCertificateMutation,
    AddVehicleRegistrationCertificateMutationVariables
  >({
    mutation: AddVehicleRegistrationCertificateDocument,
    variables: {
      input: {
        type: DocumentType.VehicleRegistrationCertificate,
        vehicleRegistrationCertificateInput: {
          images,
          issueDate: document.issueDate,
          docNumber: document.docNumber,
          lastName: document.lastName,
          lastNameLat: document.lastNameLat,
          firstName: document.firstName,
          firstNameLat: document.firstNameLat,
          middleName: document.middleName,
          registrationAddress: document.registrationAddress,
          issuerSubdivisionCode: document.issuerSubdivisionCode,
          additionalInformation: document.additionalInformation,
          carNumber: document.carNumber,
          vin: document.vin,
          model: document.model,
          modelLat: document.modelLat,
          vehicleType: document.vehicleType,
          category: document.category,
          year: document.year,
          shasiNumber: document.shasiNumber,
          bodyNumber: document.bodyNumber,
          color: document.color,
          power: document.power,
          ecoClass: document.ecoClass,
          maxWeight: document.maxWeight,
          withoutLoadWeight: document.withoutLoadWeight,
          temporaryReg: document.temporaryReg,
          pts: document.pts,
        },
      },
    },
  });

  if (!result.data || result.data.addDocument.__typename === 'AddDocumentProblem') {
    return {
      status: 'error',
      reason: AddDocumentProblemReason.Internal,
    };
  }

  return {
    status: 'ok',
  };
}

async function updateVehicleRegistrationCertificate(
  document: Omit<VehicleRegistrationCertificateDocument, 'section' | 'type'>,
) {
  const apollo = getOrCreateClient({});

  const images: DocumentImageInput[] = document.images.map((image) => ({
    id: image.id,
    originalHeight: image.originalHeight,
    originalUri: image.originalUri,
    originalUrl: image.originalUrl,
    originalWidth: image.originalWidth,
    previewHeight: image.previewHeight,
    previewUri: image.previewUri,
    previewUrl: image.previewUrl,
    previewWidth: image.previewWidth,
  }));

  const result = await apollo.mutate<
    UpdateVehicleRegistrationCertificateMutation,
    UpdateVehicleRegistrationCertificateMutationVariables
  >({
    mutation: UpdateVehicleRegistrationCertificateDocument,
    variables: {
      input: {
        type: DocumentType.VehicleRegistrationCertificate,
        vehicleRegistrationCertificateInput: {
          id: document.id,
          verificationStatus: document.verificationStatus,
          images,
          issueDate: document.issueDate,
          docNumber: document.docNumber,
          lastName: document.lastName,
          lastNameLat: document.lastNameLat,
          firstName: document.firstName,
          firstNameLat: document.firstNameLat,
          middleName: document.middleName,
          registrationAddress: document.registrationAddress,
          issuerSubdivisionCode: document.issuerSubdivisionCode,
          additionalInformation: document.additionalInformation,
          carNumber: document.carNumber,
          vin: document.vin,
          model: document.model,
          modelLat: document.modelLat,
          vehicleType: document.vehicleType,
          category: document.category,
          year: document.year,
          shasiNumber: document.shasiNumber,
          bodyNumber: document.bodyNumber,
          color: document.color,
          power: document.power,
          ecoClass: document.ecoClass,
          maxWeight: document.maxWeight,
          withoutLoadWeight: document.withoutLoadWeight,
          temporaryReg: document.temporaryReg,
          pts: document.pts,
        },
      },
      id: document.id,
    },
  });

  if (!result.data || result.data.updateDocument.__typename === 'UpdateDocumentProblem') {
    return {
      status: 'error',
      reason: UpdateDocumentProblemReason.Internal,
    };
  }

  return {
    status: 'ok',
    data:
      result.data.updateDocument.__typename === 'UpdateDocumentPayload' &&
      result.data.updateDocument.query.document,
  };
}

async function uploadImage<T>(file: File, id?: string, save = true) {
  const form = new FormData();

  form.append('file', file);
  form.append('type', DocumentType.VehicleRegistrationCertificate);
  form.append('save', save ? 'true' : 'false');

  if (id) {
    form.append('id', id);
  }

  const response = await request<T>({
    url: `/documents/upload`,
    method: 'POST',
    data: form,
  });

  return response;
}

async function uploadImages<T>(files: FileList, id: string, save = true) {
  const uploadRequests = Object.values(files).reduce<Promise<AxiosResponse<T, any>>[]>(
    (acc, file) => {
      if (file.size < MAX_FILE_SIZE_BYTES) {
        const uploadDocumentData = uploadImage<T>(file, id, save);

        acc.push(uploadDocumentData);
      }

      return acc;
    },
    [],
  );

  return await Promise.all(uploadRequests);
}

async function shareImages(images: DocumentImage[]) {
  const files: File[] = [];

  for (let i = 0; i < images.length; i++) {
    const { originalUrl } = images[i];
    const path = new URL(originalUrl).pathname;
    const response = await request({
      url: 'avatars-proxy',
      params: { path },
      responseType: 'arraybuffer',
    });

    const type = response.headers['content-type'];
    const [_, extension] = type.split('/');

    files.push(new File([new Uint8Array(response.data)], `image${i + 1}.${extension}`, { type }));
  }

  // Надо оборачивать в трайкетч, иначе отмена шаринга бросает исключение
  try {
    await navigator.share({
      files,
    });
  } catch (e) {}
}
