import cx from 'classnames';
import React, { FC } from 'react';

import { DocumentType } from '@client/shared/api/graphql';
import { useDevice } from '@client/shared/hooks';
import { DocumentsPlus } from '@client/shared/icons';
import { PressProps, mergeProps, useHover, usePress } from '@use-platform/react';
import { Text } from '@yandex-id/components';

import { i18n } from './i18n';

import styles from './Tile.module.css';

export interface ITileProps extends PressProps {
  className?: string;
  type: DocumentType;
  disabled?: boolean;
}

const getTitleByType = (type: DocumentType) => {
  switch (type) {
    case DocumentType.NationalId:
      return i18n('Паспорт РФ', { context: 'Внутренний паспорт' });
    case DocumentType.DriverLicense:
      return i18n('ВУ', { context: 'Водительское удостоверение' });
    case DocumentType.InternationalId:
      return i18n('Загран паспорт', { context: 'Заграничный паспорт' });
  }
};

/**
 * Только Паспорт, ВУ, Загран
 * @returns {JSX.Element}
 */
export const Tile: FC<ITileProps> = ({ className, type, ...restProps }) => {
  const { pressProps } = usePress(restProps);
  const { isHovered, hoverProps } = useHover({ disabled: restProps.disabled });
  const { isTouch } = useDevice();

  return (
    <div
      className={cx(className, styles.root, {
        [styles.root_isHovered]: isHovered,
      })}
      data-type={type}
      {...mergeProps(pressProps, hoverProps)}
    >
      <Text className={styles.title} variant={isTouch ? 'text-s' : 'text-m'}>
        {getTitleByType(type)}
      </Text>
      <DocumentsPlus className={styles.plus} type={type} size={isTouch ? 20 : 24} />
    </div>
  );
};
