import { useEvent, useStore } from 'effector-react';
import React, { SyntheticEvent, VFC, useCallback, useMemo, useRef, useState } from 'react';

import { Check, Close } from '@client/shared/icons';
import { ImagePresenter } from '@client/shared/ui-kit';
import { CropImage, Rect } from '@client/shared/ui-kit';
import { Button } from '@yandex-id/components/next';

import { i18n } from './i18n';
import * as model from './model/avatar-crop-modal';
import * as personalDataModalModel from './model/personal-data-modal';

import styles from './AvatarCropModal.module.css';

const INITIAL_SELECTION_SIZE = 264;
const MINIMAL_SELECTION_SIZE = 100;
const MINIMAL_SELECTION_PADDING = 13;

export const AvatarCropModal: VFC = () => {
  const isVisible = useStore(model.$isVisible);
  const uncroppedAvatar = useStore(personalDataModalModel.$uncroppedAvatar);
  const [crop, setCrop] = useState<Rect>();
  const minimalSelectionSizeRef = useRef(0);
  const htmlImageElementRef = useRef<HTMLImageElement>();
  const handleOnClose = useEvent(model.hide);

  const imageSrc = useMemo(
    () => (uncroppedAvatar ? URL.createObjectURL(uncroppedAvatar) : null),
    [uncroppedAvatar],
  );

  const handleUpload = useCallback(() => {
    if (!htmlImageElementRef.current || !crop) {
      return;
    }

    model.cropAvatar({
      image: htmlImageElementRef.current,
      rect: crop,
      type: uncroppedAvatar?.type,
    });
  }, [crop, htmlImageElementRef.current]);

  const handleLoad = useCallback(
    (event: SyntheticEvent<HTMLImageElement>) => {
      const { height, width } = event.currentTarget.getBoundingClientRect();
      const { naturalWidth, naturalHeight } = event.currentTarget;

      htmlImageElementRef.current = event.currentTarget;

      const scale = Math.min(height / naturalHeight, width / naturalWidth);
      const selectionSize =
        Math.min(
          height - MINIMAL_SELECTION_PADDING * 2,
          width - MINIMAL_SELECTION_PADDING * 2,
          INITIAL_SELECTION_SIZE,
        ) / scale;

      setCrop({
        width: selectionSize,
        height: selectionSize,
        top: naturalHeight / 2 - selectionSize / 2,
        left: naturalWidth / 2 - selectionSize / 2,
      });
      minimalSelectionSizeRef.current = MINIMAL_SELECTION_SIZE / scale;
    },
    [setCrop],
  );

  if (!imageSrc) {
    return null;
  }

  return (
    <ImagePresenter
      visible={isVisible}
      keepMounted={false}
      className={styles.root}
      onClose={handleOnClose}
    >
      <ImagePresenter.Image>
        <CropImage
          src={imageSrc}
          crop={crop}
          circularShape
          onLoad={handleLoad}
          aspectRatio={1}
          keepSelection
          minWidth={minimalSelectionSizeRef.current}
          minHeight={minimalSelectionSizeRef.current}
          className={styles.cropper}
          onCropComplete={setCrop}
        />
      </ImagePresenter.Image>
      <ImagePresenter.BottomControls>
        <Button variant="default" before={<Close />} size="xl" onPress={handleOnClose}>
          {i18n('Отмена', { context: 'Текст кнопки отмены в окне обрезания аватарки' })}
        </Button>
        <Button variant="action" before={<Check />} size="xl" onPress={handleUpload}>
          {i18n('Готово', { context: 'Текст на кнопке подтверждения обрезания аватарки' })}
        </Button>
      </ImagePresenter.BottomControls>
    </ImagePresenter>
  );
};
