import cx from 'classnames';
import { useEvent, useStore } from 'effector-react';
import { ChangeEvent, VFC, useCallback, useEffect, useRef } from 'react';

import { ProfileFieldValidationResultCode } from '@client/shared/api/graphql';
import { usePersonalInfoFormDataQuery } from '@client/shared/api/graphql';
import { CameraOutline, CameraSmall, TrashOutline } from '@client/shared/icons';
import { useForm } from '@client/shared/libs/effector-forms';
import { useAvatarsPath } from '@client/shared/paths';
import {
  Button,
  DateField,
  Form,
  Modal,
  RadioButton,
  Section,
  SectionStack,
  Select,
  Text,
  TextField,
} from '@client/shared/ui-kit';
import { mergeProps, useHover, usePress } from '@use-platform/react';

import { usePersonalInfoFormData } from './api';
import avatarPlaceholder from './assets/fill.png';
import * as model from './model/personal-data-modal';
import { i18n } from './i18n';

import styles from './PersonalDataModal.module.css';

export const PersonalDataModal: VFC = () => {
  const isVisible = useStore(model.$isVisible);
  const isPending = useStore(model.$isPending);
  const croppedAvatarUrl = useStore(model.$croppedAvatarUrl);
  const { fields, submit, setForm } = useForm(model.updatePersonalDataForm);
  const isAvatarMarkedForDelete = useStore(model.$isAvatarMarkedForDelete);
  const avatarsHost = useAvatarsPath();
  const avatarInputRef = useRef<HTMLInputElement>(null);
  const { loading, error, data: formData } = usePersonalInfoFormDataQuery();
  const isFormDataLoadingOrError = loading || error || !formData;

  const handleOnClose = useEvent(model.hide);

  useEffect(() => {
    if (!isFormDataLoadingOrError) {
      const { viewer } = formData;

      setForm({
        firstname: viewer.firstName ?? '',
        lastname: viewer.lastName ?? '',
        birthday: viewer.birthdate ? new Date(viewer.birthdate) : null,
        gender: viewer.gender ?? 'u',
        country: viewer.country ?? 'ru',
        city: viewer.city ?? '',
        timezone: viewer.timezone ?? 'Europe/Moscow',
        displayName: viewer.displayName ?? '',
      });
    }
  }, [isFormDataLoadingOrError]);

  const { isPressed: avatarBtnIsPressed, pressProps: avatarBtnPressProps } = usePress({});
  const { isHovered: avatarBtnIsHovered, hoverProps: avatarBtnHoverProps } = useHover({});

  const { isPressed: deleteAvatarBtnIsPressed, pressProps: deleteAvatarBtnPressProps } = usePress(
    {},
  );
  const { isHovered: deleteAvatarBtnIsHovered, hoverProps: deleteAvatarBtnHoverProps } = useHover(
    {},
  );

  const handleAvatarBtnClick = useCallback(() => {
    avatarInputRef.current?.click();
  }, []);

  const onAvatarChange = useCallback(
    (event: ChangeEvent<HTMLInputElement>) => {
      fields.avatar.onChange(event.target.files?.[0] ?? null);
    },
    [fields.avatar.onChange],
  );

  const handleSubmit = useCallback(() => {
    submit();
  }, [submit]);

  const { isLoading: isDataLoading, data } = usePersonalInfoFormData();
  const onGenderFieldChange = useCallback(
    (event: ChangeEvent<HTMLInputElement>) => {
      fields.gender.onChange(event.target.value);
    },
    [fields.gender.onChange],
  );

  const makeAvatarUrl = useCallback(() => {
    if (isAvatarMarkedForDelete) {
      return null;
    }

    if (croppedAvatarUrl) {
      return croppedAvatarUrl as string;
    }

    if (data?.avatarId === '0/0-0') {
      return null;
    }

    return `${avatarsHost}/get-yapic/${data?.avatarId}/islands-200`;
  }, [isAvatarMarkedForDelete, croppedAvatarUrl, data?.avatarId, avatarsHost]);

  if (isFormDataLoadingOrError || isDataLoading) {
    return null;
  }

  const {
    locationOptions: { countries, timezones },
  } = formData;

  const avatarUrl = makeAvatarUrl();

  return (
    <Modal visible={isVisible} onClose={handleOnClose} keepMounted={false} className={styles.root}>
      <Form onSubmit={handleSubmit} className={styles.form}>
        <SectionStack>
          <Section className={styles.section}>
            <Section.Title isHeading>
              {i18n('Ваши данные', {
                context: 'Заголовок диалогового окна изменения персональных данных',
              })}
            </Section.Title>
            {avatarUrl && (
              <div className={styles.avatarRow}>
                <div
                  className={cx(styles.buttonGroup, {
                    [styles.buttonGroup_isHovered]: avatarBtnIsHovered,
                    [styles.buttonGroup_isPressed]: avatarBtnIsPressed,
                  })}
                  {...mergeProps(avatarBtnHoverProps, avatarBtnPressProps)}
                >
                  <Button
                    className={styles.roundButton}
                    size="xl"
                    aria-label="Изменить фото"
                    onPress={handleAvatarBtnClick}
                  >
                    <CameraOutline />
                  </Button>
                  <input
                    type="file"
                    onChange={onAvatarChange}
                    accept=".png,.jpg,.gif,.jpeg,.bmp"
                    className={styles.avatarInput}
                    ref={avatarInputRef}
                  />
                  <div className={styles.buttonLabel} onClick={handleAvatarBtnClick}>
                    {i18n('Изменить', { context: 'Надпись на кнопке изменения аватарки' })}
                  </div>
                </div>
                <img src={avatarUrl} className={styles.avatarContainer} />
                <div
                  className={cx(styles.buttonGroup, {
                    [styles.buttonGroup_isHovered]: deleteAvatarBtnIsHovered,
                    [styles.buttonGroup_isPressed]: deleteAvatarBtnIsPressed,
                  })}
                  {...mergeProps(deleteAvatarBtnHoverProps, deleteAvatarBtnPressProps)}
                >
                  <Button
                    className={styles.roundButton}
                    size="xl"
                    aria-label="Удалить фото"
                    onPress={() => model.deleteAvatar(data?.avatarId ?? '')}
                  >
                    <TrashOutline />
                  </Button>
                  <div className={styles.buttonLabel}>
                    {i18n('Удалить', { context: 'Надпись на кнопке удаления аватарки' })}
                  </div>
                </div>
              </div>
            )}
            {!avatarUrl && (
              <div className={cx(styles.noAvatarContainer, styles.innerSection)}>
                <img src={avatarPlaceholder.src} className={styles.avatarPlaceholder} />
                <Button
                  variant="default"
                  before={<CameraSmall />}
                  className={styles.avatarBtn}
                  onPress={handleAvatarBtnClick}
                >
                  {i18n('Добавить фото', { context: 'Надпись на кнопке добавления аватарки' })}
                </Button>
                <input
                  type="file"
                  className={styles.avatarInput}
                  onChange={onAvatarChange}
                  accept=".png,.jpg,.gif,.jpeg,.bmp"
                  ref={avatarInputRef}
                />
              </div>
            )}
            <div className={styles.innerSection}>
              <TextField
                onChangeValue={fields.displayName.onChange}
                placeholder={i18n('Публичное имя', {
                  context: 'Плейсхолдер поля публичного имени',
                })}
                value={fields.displayName.value}
                size="xl"
                label={i18n('Как к вам обращаться?', { context: 'Лейбл поля публичного имени' })}
                error={fields.displayName.errorText({
                  required: i18n('Пожалуйста, укажите имя', {
                    context: 'Сообщения об ошибке при пустом публичном имени',
                  }),
                  [ProfileFieldValidationResultCode.Invalid]: i18n('Неподходящее имя.', {
                    context: 'Сообщение об ошибке при неподходящем публичном имени',
                  }),
                  [ProfileFieldValidationResultCode.Internal]: i18n(
                    'Ошибка валидации. Повторите ваш запрос позже.',
                    { context: 'Сообщение при серверной ошибке в процессе валидации' },
                  ),
                })}
              />
              <Text variant="text-s" color="secondary">
                {i18n('Это имя отображается в ваших отзывах и комментариях', {
                  context: 'Пояснение того, что собой представляет публичное имя',
                })}
              </Text>
            </div>
          </Section>
          <Section>
            <Section.Title anchor="personal-data-form-personal-data">
              {i18n('Персональные данные', {
                context:
                  'Заголовок раздела с персональными данными в форме редактирования персональных и публичных данных',
              })}
            </Section.Title>
            <div className={styles.innerSection}>
              <TextField
                onChangeValue={fields.firstname.onChange}
                placeholder={i18n('Имя', { context: 'Плейсхолдер поля с именем' })}
                size="xl"
                value={fields.firstname.value}
                label={i18n('Имя и фамилия', { context: 'Лейбл для полей имени и фамилии' })}
                error={fields.firstname.errorText({
                  required: i18n('Пожалуйста, укажите имя', {
                    context: 'Сообщения об ошибке при пустом имени',
                  }),
                  [ProfileFieldValidationResultCode.Invalid]: i18n('Неподходящее имя.', {
                    context: 'Сообщение об ошибке при неподходящем имени',
                  }),
                  [ProfileFieldValidationResultCode.Internal]: i18n(
                    'Ошибка валидации. Повторите ваш запрос позже.',
                    { context: 'Сообщение при серверной ошибке в процессе валидации' },
                  ),
                })}
              />
              <TextField
                onChangeValue={fields.lastname.onChange}
                placeholder={i18n('Фамилия', { context: 'Плейсхолдер поля с фамилией' })}
                size="xl"
                value={fields.lastname.value}
                error={fields.lastname.errorText({
                  required: i18n('Пожалуйста, укажите фамилию', {
                    context: 'Сообщения об ошибке при пустой фамилии',
                  }),
                  [ProfileFieldValidationResultCode.Invalid]: i18n('Неподходящяя фамилия.', {
                    context: 'Сообщение об ошибке при неподходящей фамилии',
                  }),
                  [ProfileFieldValidationResultCode.Internal]: i18n(
                    'Ошибка валидации. Повторите ваш запрос позже.',
                    { context: 'Сообщение при серверной ошибке в процессе валидации' },
                  ),
                })}
              />
            </div>
            <div className={cx(styles.innerSection, styles.genderAndBirthdaySection)}>
              <Text variant="text-m-long" className={styles.genderLabel}>
                {i18n('Пол', { context: 'Лейбл для поля пола' })}
              </Text>
              <RadioButton
                options={[
                  { value: 'm', children: i18n('М', { context: 'Сокращение для мужского пола' }) },
                  { value: 'f', children: i18n('Ж', { context: 'Сокращение для женского пола' }) },
                ]}
                value={fields.gender.value}
                className={styles.genderField}
                onChange={onGenderFieldChange}
                size="xl"
              />
              <DateField
                label={i18n('Дата рождения', { context: 'Лейбл для поля дня рождения' })}
                className={styles.birthdayField}
                onChangeValue={fields.birthday.onChange}
                value={fields.birthday.value}
                size="xl"
              />
            </div>
          </Section>
          <Section>
            <div className={styles.innerSection}>
              <Text variant="text-m-long">
                {i18n('Страна и населенный пункт', {
                  context: 'Лейбл для полей страны и населенного пункта',
                })}
              </Text>
              <Select onChange={fields.country.onChange} value={fields.country.value}>
                {countries?.map((country) => (
                  <Select.Option key={country.code} value={country.code}>
                    {country.name}
                  </Select.Option>
                ))}
              </Select>
              <TextField
                onChangeValue={fields.city.onChange}
                placeholder={i18n('Населенный пункт', {
                  context: 'Плейсхолдер для поля населенного пункта',
                })}
                value={fields.city.value}
              />
            </div>
            <div className={styles.innerSection}>
              <Text variant="text-m-long">
                {i18n('Часовой пояс', { context: 'Лейбл для поля часового пояса' })}
              </Text>
              <Select onChange={fields.timezone.onChange} value={fields.timezone.value}>
                {timezones?.map((timezone) => (
                  <Select.Option key={timezone.code} value={timezone.code}>
                    {timezone.name}
                  </Select.Option>
                ))}
              </Select>
            </div>
            <Button
              type="submit"
              size="xl"
              variant="action"
              onPress={handleSubmit}
              disabled={isPending}
              className={styles.action}
            >
              {i18n('Сохранить', {
                context: 'Текст на кнопке сохранения формы персональных и публичных данных',
              })}
            </Button>
          </Section>
        </SectionStack>
      </Form>
    </Modal>
  );
};
