import Axios from 'axios';
import { createEffect } from 'effector';

import {
  ProfileFieldName,
  ProfileFieldValidationResultCode,
  UpdatePersonalInfoDocument,
  UpdatePersonalInfoInput,
  UpdatePersonalInfoMutation,
  UpdatePersonalInfoMutationVariables,
  ValidatePersonalInfoFieldDocument,
  ValidatePersonalInfoFieldMutation,
  ValidatePersonalInfoFieldMutationVariables,
  usePersonalInfoFormDataQuery,
} from '@client/shared/api/graphql';
import { getOrCreateClient, useMappedQuery } from '@client/shared/libs/apollo';

export async function updatePersonalInfo(
  input: UpdatePersonalInfoInput & { avatar: Blob | null; avatarFileName: string | null },
) {
  const { avatar, avatarFileName, ...graphqlInput } = input;
  const apollo = getOrCreateClient({});

  if (avatar) {
    const { request } = Axios.create({
      baseURL: '/personal/api/',
      timeout: 10000,
    });
    const form = new FormData();

    form.append('file', avatar, avatarFileName ?? '');
    const avatarResult = await request<{ status: 'error' | 'ok'; errors: string[] }>({
      url: '/viewer/avatar-upload',
      method: 'POST',
      data: form,
    });

    if (avatarResult.data.status === 'error') {
      return Promise.reject({ name: avatarResult.data.errors[0] });
    }
  }

  const result = await apollo.mutate<
    UpdatePersonalInfoMutation,
    UpdatePersonalInfoMutationVariables
  >({
    mutation: UpdatePersonalInfoDocument,
    variables: { input: graphqlInput },
  });

  if (result.data?.viewerUpdatePersonalInfo.__typename === 'UpdatePersonalInfoProblem') {
    return Promise.reject({ name: result.data?.viewerUpdatePersonalInfo.reason });
  }

  return true;
}

export async function validatePersonalInfo(fields: Record<ProfileFieldName, string>) {
  const apollo = getOrCreateClient({});
  const results: Record<ProfileFieldName, ProfileFieldValidationResultCode> = {
    [ProfileFieldName.DisplayName]: ProfileFieldValidationResultCode.Internal,
    [ProfileFieldName.Firstname]: ProfileFieldValidationResultCode.Internal,
    [ProfileFieldName.Lastname]: ProfileFieldValidationResultCode.Internal,
  };

  for (const fieldName of Object.keys(fields) as ProfileFieldName[]) {
    const result = await apollo.mutate<
      ValidatePersonalInfoFieldMutation,
      ValidatePersonalInfoFieldMutationVariables
    >({
      mutation: ValidatePersonalInfoFieldDocument,
      variables: { input: { fieldName, fieldValue: fields[fieldName] } },
    });

    results[fieldName] =
      result.data?.viewerValidatePersonalInfoField.code ??
      ProfileFieldValidationResultCode.Internal;
  }

  if (Object.values(results).some((val) => val !== ProfileFieldValidationResultCode.Ok)) {
    return Promise.reject(results);
  }

  return results;
}

export function usePersonalInfoFormData() {
  return useMappedQuery(usePersonalInfoFormDataQuery, (data) => {
    const {
      viewer: { avatarId },
    } = data;

    return { avatarId };
  });
}

export const updatePersonalInfoApiFx = createEffect(updatePersonalInfo);

export const validatePersonalInfoFx = createEffect<
  Record<ProfileFieldName, string>,
  Record<ProfileFieldName, ProfileFieldValidationResultCode>,
  Record<ProfileFieldName, ProfileFieldValidationResultCode>
>(validatePersonalInfo);
