import {
  Store,
  attach,
  createEffect,
  createEvent,
  createStore,
  forward,
  guard,
  merge,
  restore,
} from 'effector';

import {
  ProfileFieldName,
  ProfileFieldValidationResultCode,
  UpdatePersonalInfoProblemKind,
} from '@client/shared/api/graphql';
import { createForm, rules } from '@client/shared/libs/effector-forms';
import { createDialogApi } from '@client/shared/libs/effector-helpers';

import { updatePersonalInfoApiFx, validatePersonalInfoFx } from '../api';
import { cropAvatarFx } from './avatar-crop-modal';
import * as avatarCropModalModel from './avatar-crop-modal';
import {
  avatarFileSizeError,
  avatarSizeError,
  deleteAvatarError,
  personalDataError,
} from './notifications';

// Форматирует дату в ГГГГ-ММ-ДД
const formatDate = (val: Date | null) =>
  val
    ? [
        String(val.getFullYear()).padStart(4, '0'),
        String(val.getMonth() + 1).padStart(2, '0'),
        String(val.getDate()).padStart(2, '0'),
      ].join('-')
    : '';

export const { show, hide, $isVisible } = createDialogApi('personal-data');
export const deleteAvatar = createEvent<string>();

export const updatePersonalDataForm = createForm({
  fields: {
    firstname: {
      init: '',
      rules: [rules.required()],
    },
    lastname: {
      init: '',
      rules: [rules.required()],
    },
    birthday: {
      init: null as Date | null,
    },
    gender: {
      init: 'UNKNOWN',
    },
    country: {
      init: 'ru',
    },
    city: {
      init: '',
    },
    timezone: {
      init: 'Europe/Moscow',
    },
    displayName: {
      init: '',
      rules: [rules.required()],
    },
    avatar: {
      init: null as File | null,
    },
  },
});

export const $croppedAvatar = restore(cropAvatarFx.doneData, null).reset(hide).reset(deleteAvatar);

export const $croppedAvatarUrl = $croppedAvatar.map((val) =>
  val ? URL.createObjectURL(val) : null,
);

export const $isAvatarMarkedForDelete = createStore(false)
  .on(deleteAvatar, () => true)
  .reset(cropAvatarFx.doneData);

const validateAvatarFx = createEffect((val: File) => {
  const fr = new FileReader();

  return new Promise<File>((resolve, reject) => {
    fr.readAsDataURL(val);
    fr.onloadend = (event) => {
      const image = new Image();

      image.src = event.target?.result as string;
      image.onload = () => {
        if (image.width < 200 || image.height < 200) {
          reject();
        } else {
          resolve(val);
        }
      };
    };
  });
});

export const $uncroppedAvatar = restore(validateAvatarFx, null);
const $originalFileName = ($uncroppedAvatar as Store<File | null>).map((val) => val?.name ?? null);

const updatePersonalInfoFx = attach({
  effect: updatePersonalInfoApiFx,
  source: {
    firstname: updatePersonalDataForm.fields.firstname.$value,
    lastname: updatePersonalDataForm.fields.lastname.$value,
    birthday: updatePersonalDataForm.fields.birthday.$value.map(formatDate),
    gender: updatePersonalDataForm.fields.gender.$value,
    country: updatePersonalDataForm.fields.country.$value,
    city: updatePersonalDataForm.fields.city.$value,
    timezone: updatePersonalDataForm.fields.timezone.$value,
    displayName: updatePersonalDataForm.fields.displayName.$value,
    avatar: $croppedAvatar,
    avatarFileName: $originalFileName,

    deleteAvatar: $isAvatarMarkedForDelete,
  },
});

validatePersonalInfoFx.failData.watch((results) => {
  if (results[ProfileFieldName.DisplayName] !== ProfileFieldValidationResultCode.Ok) {
    updatePersonalDataForm.fields.displayName.addError({
      rule: results[ProfileFieldName.DisplayName],
    });
  }

  if (results[ProfileFieldName.Firstname] !== ProfileFieldValidationResultCode.Ok) {
    updatePersonalDataForm.fields.firstname.addError({
      rule: results[ProfileFieldName.Firstname],
    });
  }

  if (results[ProfileFieldName.Lastname] !== ProfileFieldValidationResultCode.Ok) {
    updatePersonalDataForm.fields.lastname.addError({
      rule: results[ProfileFieldName.Lastname],
    });
  }
});

export const $isPending = restore(
  merge([updatePersonalInfoFx.pending, validatePersonalInfoFx.pending]),
  false,
);

guard({
  clock: updatePersonalDataForm.fields.avatar.changed,
  source: updatePersonalDataForm.fields.avatar.$value as Store<File>,
  filter: updatePersonalDataForm.fields.avatar.$value.map((val) => val !== null),
  target: validateAvatarFx,
});

forward({ from: cropAvatarFx.doneData, to: show });
forward({ from: validateAvatarFx.failData, to: avatarSizeError });
forward({ from: validateAvatarFx.doneData, to: [avatarCropModalModel.show, hide] });

forward({
  from: updatePersonalDataForm.formValidated.map((val) => ({
    [ProfileFieldName.DisplayName]: val.displayName,
    [ProfileFieldName.Firstname]: val.firstname,
    [ProfileFieldName.Lastname]: val.lastname,
  })),
  to: validatePersonalInfoFx,
});

forward({
  from: validatePersonalInfoFx.doneData,
  to: updatePersonalInfoFx,
});

guard({
  source: updatePersonalInfoFx.failData,
  filter: (val) => val.name === UpdatePersonalInfoProblemKind.UpdateDataInternal,
  target: personalDataError,
});

guard({
  source: updatePersonalInfoFx.failData,
  filter: (val) => val.name === UpdatePersonalInfoProblemKind.DeleteAvatarInternal,
  target: deleteAvatarError,
});

guard({
  source: updatePersonalInfoFx.failData,
  filter: (val) => val.name === 'change_avatar.invalid_image_size',
  target: avatarSizeError,
});

guard({
  source: updatePersonalInfoFx.failData,
  filter: (val) => val.name === 'change_avatar.invalid_file_size',
  target: avatarFileSizeError,
});

forward({ from: updatePersonalInfoFx.doneData, to: hide });
forward({ from: avatarCropModalModel.hide, to: show });
