import { useStore } from 'effector-react';
import { VFC, useCallback } from 'react';

import { useForm } from '@client/shared/libs/effector-forms';
import { useEnvContext } from '@client/shared/libs/env';
import {
  Button,
  Example,
  Form,
  Modal,
  Section,
  SectionStack,
  TextField,
} from '@client/shared/ui-kit';

import { i18n, i18nRaw } from './i18n';
import * as model from './model';

import styles from './PublicIdModal.module.css';

const publicIdErrorMap = (tld: string) => ({
  'public_id.empty': i18n('Вы не ввели адрес'),
  'public_id.not_available': i18n('Этот адрес занят'),
  'public_id.long': i18n('Адрес слишком длинный'),
  'public_id.startswithdigit': i18n('Адрес не может начинаться с цифры'),
  'public_id.startswithdot': i18n('Адрес не может начинаться с точки'),
  'public_id.startswithhyphen': i18n('Адрес не может начинаться с дефиса'),
  'public_id.endwithdot': i18n('Адрес не может заканчиваться точкой'),
  'public_id.endswithhyphen': i18n('Адрес не может заканчиваться дефисом'),
  'public_id.doubleddot': i18n('Пожалуйста, не используйте две точки подряд'),
  'public_id.doubledhyphen': i18n('Пожалуйста, не используйте два дефиса подряд'),
  'public_id.prohibitedsymbols': i18n('В адресе можно использовать только цифры и латинские буквы'),
  'public_id.dothyphen': i18n('Пожалуйста, не ставьте точку и дефис подряд'),
  'public_id.hyphendot': i18n('Пожалуйста, не ставьте точку и дефис подряд'),
  'public_id.update_not_allowed': i18nRaw(
    'Вы уже меняли адрес дважды. Если нужно изменить его ещё раз, напишите в {support}',
    {
      support: (
        <a href={`https://yandex.${tld}/support/passport/public-profile.html`}>
          {i18n('службу поддержки', {
            context: 'Часть фразы "напишите в службу поддержки"',
          })}
        </a>
      ),
    },
  ) as unknown as string,
  'public_id.invalid': i18n('Введенное значение некорректно'),
});

const DEFAULT_PUBLIC_ID = 'yourname';

export const PublicIdModal: VFC = () => {
  const { tld } = useEnvContext();
  const { fields, submit } = useForm(model.setPublicIdForm);

  const handleSubmit = useCallback(() => {
    submit();
  }, [submit]);

  const handleOnClose = useCallback(() => {
    model.hide();
  }, []);

  const isLoading = useStore(model.$isLoading);
  const viewerData = useStore(model.$viewerData);
  const isVisible = useStore(model.$isVisible);
  const isPending = useStore(model.$isPending);
  const suggestedPublicIds = useStore(model.$suggestedPublicId);
  const isValidPublicId = useStore(model.$isPublicIdValid);

  if (isLoading) {
    return null;
  }

  const setSuggestedId = (id: string) => () => {
    fields.publicId.onChange(id);
  };

  const userPublicId =
    viewerData?.viewer.hasPublicIdSet && viewerData.viewer.publicId
      ? viewerData.viewer.publicId
      : DEFAULT_PUBLIC_ID;

  return (
    <Modal visible={isVisible} onClose={handleOnClose} keepMounted={false} className={styles.modal}>
      <Form onSubmit={handleSubmit}>
        <SectionStack>
          <Section>
            <Section.Title isHeading>{i18n('Ссылки на ваши публичные страницы')}</Section.Title>
            <Section.Description>
              {i18n(
                'Придумайте удобный запоминающийся адрес для ваших публичных страниц на Яндексе',
              )}
            </Section.Description>
            <Example type="market" id={fields.publicId.value || userPublicId} />
            <Example type="q" id={fields.publicId.value || userPublicId} />

            <TextField
              autoFocus={isVisible}
              placeholder={userPublicId}
              label={i18n('Придумайте адрес (на латинице)')}
              value={fields.publicId.value}
              onChangeValue={fields.publicId.onChange}
              maxLength={30}
              error={fields.publicId.errorText(publicIdErrorMap(tld))}
            />

            <div className={styles.suggestWrapper}>
              {suggestedPublicIds.map((id, index) => {
                return (
                  <Button
                    key={index}
                    className={styles.suggest}
                    type="button"
                    size="m"
                    variant="default"
                    onPress={setSuggestedId(id)}
                  >
                    {id}
                  </Button>
                );
              })}
            </div>

            <Button
              type="submit"
              size="l"
              variant="action"
              onPress={handleSubmit}
              disabled={!isValidPublicId || isPending}
            >
              {i18n('Сохранить')}
            </Button>
          </Section>
        </SectionStack>
      </Form>
    </Modal>
  );
};
