import { attach, combine, createEffect, createStore, forward, guard, restore } from 'effector';

import { PersonalPublicIdQuery, PublicIdUpdateInput } from '@client/shared/api/graphql';
import { canUseDom } from '@client/shared/libs/can-use-dom';
import { createForm } from '@client/shared/libs/effector-forms';
import {
  $dialog,
  hideDialog,
  showDialog,
} from '@client/shared/libs/effector-helpers/persist-dialog';

import { getViewerData, suggestPublicId, updatePublicId, validatePublicId } from './api';

const $abortController = createStore(canUseDom ? new AbortController() : null);

// Form
export const setPublicIdForm = createForm({
  fields: {
    publicId: {
      init: '',
      rules: [
        {
          name: 'required',
          validator: (value) => Boolean(value) && value.length > 0 && value.length < 30,
        },
      ],
    },
  },
  validateOn: ['blur', 'change', 'submit'],
});

// Modal
const DIALOG_KEY = 'public-id';

export const show = showDialog.prepend(() => ({
  key: DIALOG_KEY,
}));
export const hide = hideDialog.prepend(() => ({
  key: DIALOG_KEY,
}));

export const $isVisible = $dialog.map((dialog) => dialog === DIALOG_KEY);

// Viewer
export const viewerDataFx = createEffect(getViewerData);

export const $viewerData = createStore<PersonalPublicIdQuery | null>(null).on(
  viewerDataFx.doneData,
  (_, viewerData) => viewerData,
);

export const $isLoading = restore(viewerDataFx.pending.updates, true);

// Suggest
export const suggestPublicIdFx = createEffect(suggestPublicId);

export const $suggestedPublicId = createStore<string[]>([]).on(
  suggestPublicIdFx.doneData,
  (_, suggested) => suggested,
);

// Validate
const validatePublicIdFx = createEffect(async (value: string) => {
  const validateResponse = await validatePublicId(
    {
      publicId: value,
    },
    $abortController.getState()?.signal,
  );

  if (validateResponse !== 'ok') {
    return Promise.reject({ reason: validateResponse });
  }

  return true;
});

validatePublicIdFx.failData.watch((problem) => {
  const { reason } = problem as unknown as { reason: string };

  setPublicIdForm.fields.publicId.addError({
    rule: reason,
  });
});

setPublicIdForm.fields.publicId.onChange.watch(() => {
  const isPendingValidate = validatePublicIdFx.pending.getState();

  if (isPendingValidate) {
    $abortController.getState()?.abort();
  }
});

$abortController.on(validatePublicIdFx, () => new AbortController());

export const $isPublicIdValid = createStore(false);

$isPublicIdValid.on(validatePublicIdFx.doneData, () => true);
$isPublicIdValid.on(validatePublicIdFx.failData, () => false);
$isPublicIdValid.on(setPublicIdForm.fields.publicId.onChange, (_, value) => value !== '');

// SetPublicId
const setPublicIdFx = attach({
  effect: createEffect(async (input: PublicIdUpdateInput) => {
    const response = await updatePublicId(input);

    if (response !== 'ok') {
      return Promise.reject({ reason: response });
    }
  }),
  source: {
    publicId: setPublicIdForm.fields.publicId.$value,
  },
});

setPublicIdFx.failData.watch((problem) => {
  const { reason } = problem as unknown as { reason: string };

  setPublicIdForm.fields.publicId.addError({
    rule: reason,
  });
});

export const $isPending = restore(setPublicIdFx.pending.updates, false);

// Actions
forward({
  from: $isVisible,
  to: viewerDataFx,
});

forward({
  from: combine({
    lastName: $viewerData.map((viewer) => viewer?.viewer.lastName),
    firstName: $viewerData.map((viewer) => viewer?.viewer.firstName),
  }),
  to: suggestPublicIdFx,
});

forward({ from: setPublicIdForm.formValidated, to: setPublicIdFx });
forward({ from: setPublicIdFx.doneData, to: hide });

guard({
  source: setPublicIdForm.fields.publicId.onChange,
  filter: setPublicIdForm.fields.publicId.$isValid,
  target: validatePublicIdFx,
});

forward({ from: hide, to: setPublicIdForm.reset });
