import { ChangeEvent, VFC, useCallback } from 'react';

import phoneFilledImageSrc from '@client/shared/assets/phone-filled.png';
import { Button, CardTemplate, Form, ImageIcon, Text, TextField } from '@client/shared/ui-kit';

import { i18n } from './i18n';
import { PhoneBindingFormEvent, PhoneBindingFormEventType } from './types';

export interface PhoneBindingFormProps {
  number?: string;
  error?: string;
  pending?: boolean;
  onAction: (event: PhoneBindingFormEvent) => void;
}

export const PhoneBindingForm: VFC<PhoneBindingFormProps> = (props) => {
  const { number = '', error, pending, onAction } = props;

  const handleChange = useCallback(
    (event: ChangeEvent<HTMLInputElement>) => {
      onAction({ type: PhoneBindingFormEventType.CHANGE_NUMBER, number: event.target.value });
    },
    [onAction],
  );

  const handleSubmit = useCallback(() => {
    onAction({ type: PhoneBindingFormEventType.SUBMIT });
  }, [onAction]);

  return (
    <Form onSubmit={handleSubmit}>
      <CardTemplate>
        <CardTemplate.Media>
          <ImageIcon src={phoneFilledImageSrc} size={120} />
        </CardTemplate.Media>
        <CardTemplate.Content>
          <Text variant="heading-m" weight="bold">
            {i18n('Телефон для защиты аккаунта')}
          </Text>
          <Text variant="text-m-long">
            {i18n('Введите телефон, на который будут приходить уведомления безопасности')}
          </Text>
        </CardTemplate.Content>

        <TextField
          type="tel"
          placeholder="+7 (000) 000-00-00"
          value={number}
          onChange={handleChange}
          error={error}
          disabled={pending}
        />

        <CardTemplate.Actions>
          <Button variant="action" size="xl" onPress={handleSubmit} disabled={pending}>
            {i18n('Далее')}
          </Button>
        </CardTemplate.Actions>
      </CardTemplate>
    </Form>
  );
};
