import { RefObject, VFC, useCallback, useImperativeHandle } from 'react';

import {
  SendCodeToBindPrimaryPhoneProblemReason,
  useSendCodeToBindPrimaryPhoneMutation,
} from '@client/shared/api/graphql';
import { FormState, useForm, useFormEvents } from '@client/shared/libs/effector-forms';

import { BindPrimaryPhoneNotifier } from '../../notifier';
import {
  PhoneBindingForm,
  PhoneBindingFormEvent,
  PhoneBindingFormEventType,
} from '../phone-binding-form';
import { PhoneBindingFormData, createPhoneBindingForm } from './form';
import { i18n } from './i18n';

export interface PhoneBindingPayload {
  trackId: string;
  number: string;
  expiryTimestamp: number;
}

export interface PhoneBindingProps {
  formRef?: RefObject<FormState<PhoneBindingFormData>>;
  onCompleted: (payload: PhoneBindingPayload) => void;
}

export const PhoneBinding: VFC<PhoneBindingProps> = (props) => {
  const { formRef, onCompleted } = props;
  const form = useForm(createPhoneBindingForm);
  const { fields, submit, reset } = form;
  const { number } = fields;

  useImperativeHandle(formRef, () => form);

  const [sendCode, { loading: pending }] = useSendCodeToBindPrimaryPhoneMutation({
    onCompleted: (data) => {
      if (data.response.__typename === 'SendCodeToBindPrimaryPhonePayload') {
        onCompleted(data.response);
        reset();
      } else {
        switch (data.response.reason) {
          case SendCodeToBindPrimaryPhoneProblemReason.PhoneNumberBlocked:
            number.addError({ rule: 'blocked' });
            break;

          case SendCodeToBindPrimaryPhoneProblemReason.PhoneNumberInvalid:
            number.addError({ rule: 'invalid' });
            break;

          case SendCodeToBindPrimaryPhoneProblemReason.SmsLimitExceeded:
            BindPrimaryPhoneNotifier.smsLimitExceeded();
            break;

          default:
            BindPrimaryPhoneNotifier.internal();
            break;
        }
      }
    },
  });

  const handleAction = useCallback(
    (event: PhoneBindingFormEvent) => {
      switch (event.type) {
        case PhoneBindingFormEventType.CHANGE_NUMBER:
          number.onChange(event.number);
          break;

        case PhoneBindingFormEventType.SUBMIT:
          BindPrimaryPhoneNotifier.hide();
          submit();
          break;
      }
    },
    [number, submit],
  );

  useFormEvents(form, {
    onSubmit: (input) => {
      sendCode({ variables: { input } });
    },
  });

  return (
    <PhoneBindingForm
      pending={pending}
      number={number.value}
      error={number.errorText({
        required: i18n('Введите телефон'),
        invalid: i18n('Не получится использовать этот номер, попробуйте ввести другой'),
        blocked: i18n('Не получится использовать этот номер, попробуйте ввести другой'),
      })}
      onAction={handleAction}
    />
  );
};
