import { RefObject, VFC, useCallback, useImperativeHandle, useState } from 'react';

import {
  SmsCodeConfirmationEvent,
  SmsCodeConfirmationEventType,
  SmsCodeConfirmationForm,
} from '@client/entities/SmsCodeConfirmationForm';
import {
  SendCodeToBindPrimaryPhoneProblemReason,
  VerifyCodeToBindPrimaryPhoneProblemReason,
  useSendCodeToBindPrimaryPhoneMutation,
  useVerifyCodeToBindPrimaryPhoneMutation,
} from '@client/shared/api/graphql';
import { FormState, useForm, useFormEvents } from '@client/shared/libs/effector-forms';

import { BindPrimaryPhoneNotifier } from '../../notifier';
import { SmsCodeConfirmationFormData, createSmsCodeConfirmationForm } from './form';

export interface SmsCodeConfirmationPayload {
  trackId: string;
}

export interface SmsCodeConfirmationProps {
  number: string;
  trackId: string;
  expiryTimestamp?: number;
  formRef?: RefObject<FormState<SmsCodeConfirmationFormData>>;
  onCompleted: (payload: SmsCodeConfirmationPayload) => void;
}

export const SmsCodeConfirmation: VFC<SmsCodeConfirmationProps> = (props) => {
  const { number, trackId, formRef, onCompleted } = props;
  const form = useForm(createSmsCodeConfirmationForm);
  const [expiryTimestamp, setExpiryTimestamp] = useState(props.expiryTimestamp);
  const { fields, submit, reset } = form;
  const { code } = fields;

  useImperativeHandle(formRef, () => form);

  const [resendCode, { loading: resendLoading }] = useSendCodeToBindPrimaryPhoneMutation({
    onCompleted: (data) => {
      if (data.response.__typename === 'SendCodeToBindPrimaryPhonePayload') {
        setExpiryTimestamp(data.response.expiryTimestamp);
      } else {
        switch (data.response.reason) {
          case SendCodeToBindPrimaryPhoneProblemReason.PhoneNumberBlocked:
            BindPrimaryPhoneNotifier.phoneNumberBlocked();
            break;

          // NOTE: На практике эта ошибка не должна возникать, так как если мы попали на этот экран,
          // то уже отправили валидный номер
          case SendCodeToBindPrimaryPhoneProblemReason.PhoneNumberInvalid:
            BindPrimaryPhoneNotifier.internal();
            break;

          case SendCodeToBindPrimaryPhoneProblemReason.SmsLimitExceeded:
            BindPrimaryPhoneNotifier.smsLimitExceeded();
            break;

          default:
            BindPrimaryPhoneNotifier.internal();
            break;
        }
      }
    },
    variables: { input: { number, trackId } },
  });

  const resend = useCallback(() => {
    reset();
    resendCode();
  }, [reset, resendCode]);

  const [verifyCode, { loading: verifyLoading }] = useVerifyCodeToBindPrimaryPhoneMutation({
    onCompleted: (data) => {
      if (data.response.__typename === 'VerifyCodeToBindPrimaryPhonePayload') {
        const { trackId } = data.response;

        onCompleted({ trackId });
        reset();
      } else {
        switch (data.response.reason) {
          case VerifyCodeToBindPrimaryPhoneProblemReason.CodeEmpty:
            code.addError({ rule: 'required' });
            break;

          case VerifyCodeToBindPrimaryPhoneProblemReason.CodeInvalid:
            code.addError({ rule: 'invalid' });
            break;

          case VerifyCodeToBindPrimaryPhoneProblemReason.CodeNotSent:
            BindPrimaryPhoneNotifier.codeNotSent();
            break;

          case VerifyCodeToBindPrimaryPhoneProblemReason.ConfirmationsLimitExceeded:
            BindPrimaryPhoneNotifier.confirmationsLimitExceeded();
            break;

          default:
            BindPrimaryPhoneNotifier.internal();
            break;
        }
      }
    },
  });

  const handleAction = useCallback(
    (event: SmsCodeConfirmationEvent) => {
      switch (event.type) {
        case SmsCodeConfirmationEventType.CHANGE_CODE:
          code.onChange(event.code);
          break;

        case SmsCodeConfirmationEventType.RESEND_CODE:
          BindPrimaryPhoneNotifier.hide();
          resend();
          break;

        case SmsCodeConfirmationEventType.VERIFY_CODE:
          BindPrimaryPhoneNotifier.hide();
          submit();
          break;
      }
    },
    [code, resend, submit],
  );

  useFormEvents(form, {
    onSubmit: (input) => {
      verifyCode({
        variables: {
          input: { ...input, trackId },
        },
      });
    },
  });

  const pending = resendLoading || verifyLoading;

  return (
    <SmsCodeConfirmationForm
      pending={pending}
      number={number}
      expiryTimestamp={expiryTimestamp}
      onAction={handleAction}
      code={fields.code.value}
      invalid={fields.code.hasError()}
    />
  );
};
