import { VFC, useCallback } from 'react';

import {
  SendCodeToConfirmPhoneProblemReason,
  useSendCodeToConfirmPhoneMutation,
} from '@client/shared/api/graphql';

import { ConfirmPhoneBySmsSnackbars } from '../../notifier';
import {
  PhoneConfirmationForm,
  PhoneConfirmationFormEvent,
  PhoneConfirmationFormEventType,
} from '../phone-confirmation-form';

export interface PhoneConfirmationPayload {
  trackId: string;
  number: string;
  expiryTimestamp: number;
}

export interface PhoneConfirmationProps {
  number: string;
  onClose: () => void;
  onCompleted: (payload: PhoneConfirmationPayload) => void;
}

export const PhoneConfirmation: VFC<PhoneConfirmationProps> = (props) => {
  const { number, onClose, onCompleted } = props;

  const [sendCode, { loading }] = useSendCodeToConfirmPhoneMutation({
    onCompleted: (data) => {
      if (data.response.__typename === 'SendCodeToConfirmPhonePayload') {
        onCompleted(data.response);
      } else {
        switch (data.response.reason) {
          case SendCodeToConfirmPhoneProblemReason.PhoneNumberBlocked:
            ConfirmPhoneBySmsSnackbars.phoneNumberBlocked();
            break;

          // NOTE: На практике это ошибка не должна возникать, так как мы подтверждаем привязанный номер
          // но на всякий случай показываем что что-то пошло не так
          case SendCodeToConfirmPhoneProblemReason.PhoneNumberInvalid:
            ConfirmPhoneBySmsSnackbars.internal();
            break;

          case SendCodeToConfirmPhoneProblemReason.SmsLimitExceeded:
            ConfirmPhoneBySmsSnackbars.smsLimitExceeded();
            break;

          default:
            ConfirmPhoneBySmsSnackbars.internal();
            break;
        }
      }
    },
    variables: { input: { number } },
  });

  const handleAction = useCallback(
    (event: PhoneConfirmationFormEvent) => {
      switch (event.type) {
        case PhoneConfirmationFormEventType.CANCEL:
          onClose();
          break;

        case PhoneConfirmationFormEventType.SEND_CODE:
          ConfirmPhoneBySmsSnackbars.hide();
          sendCode();
          break;
      }
    },
    [onClose, sendCode],
  );

  return <PhoneConfirmationForm loading={loading} number={number} onAction={handleAction} />;
};
