import { RefObject, VFC, useCallback, useImperativeHandle, useState } from 'react';

import {
  SmsCodeConfirmationEvent,
  SmsCodeConfirmationEventType,
  SmsCodeConfirmationForm,
} from '@client/entities/SmsCodeConfirmationForm';
import {
  SendCodeToConfirmPhoneProblemReason,
  VerifyCodeToConfirmPhoneProblemReason,
  useSendCodeToConfirmPhoneMutation,
  useVerifyCodeToConfirmPhoneMutation,
} from '@client/shared/api/graphql';
import { FormState, useForm, useFormEvents } from '@client/shared/libs/effector-forms';

import { ConfirmPhoneBySmsSnackbars } from '../../notifier';
import { SmsCodeConfirmationFormData, createSmsCodeConfirmationForm } from './form';

export interface SmsCodeConfirmationPayload {
  trackId: string;
}

export interface SmsCodeConfirmationProps {
  number: string;
  trackId: string;
  expiryTimestamp?: number;
  formRef?: RefObject<FormState<SmsCodeConfirmationFormData>>;
  onCompleted: (payload: SmsCodeConfirmationPayload) => void;
}

export const SmsCodeConfirmation: VFC<SmsCodeConfirmationProps> = (props) => {
  const { number, trackId, formRef, onCompleted } = props;
  const form = useForm(createSmsCodeConfirmationForm);
  const [expiryTimestamp, setExpiryTimestamp] = useState(props.expiryTimestamp);
  const { fields, submit, reset } = form;
  const { code } = fields;

  useImperativeHandle(formRef, () => form);

  const [resendCode, { loading: resendLoading }] = useSendCodeToConfirmPhoneMutation({
    onCompleted: (data) => {
      if (data.response.__typename === 'SendCodeToConfirmPhonePayload') {
        setExpiryTimestamp(data.response.expiryTimestamp);
      } else {
        switch (data.response.reason) {
          case SendCodeToConfirmPhoneProblemReason.PhoneNumberBlocked:
            ConfirmPhoneBySmsSnackbars.phoneNumberBlocked();
            break;

          // NOTE: На практике это ошибка не должна возникать, так как мы подтверждаем привязанный номер
          // но на всякий случай показываем что что-то пошло не так
          case SendCodeToConfirmPhoneProblemReason.PhoneNumberInvalid:
            ConfirmPhoneBySmsSnackbars.internal();
            break;

          case SendCodeToConfirmPhoneProblemReason.SmsLimitExceeded:
            ConfirmPhoneBySmsSnackbars.smsLimitExceeded();
            break;

          default:
            ConfirmPhoneBySmsSnackbars.internal();
            break;
        }
      }
    },
    variables: { input: { number, trackId } },
  });

  const resend = useCallback(() => {
    reset();
    resendCode();
  }, [reset, resendCode]);

  const [verifyCode, { loading: verifyLoading }] = useVerifyCodeToConfirmPhoneMutation({
    onCompleted: (data) => {
      if (data.response.__typename === 'VerifyCodeToConfirmPhonePayload') {
        const { trackId } = data.response;

        onCompleted({ trackId });
        reset();
      } else {
        switch (data.response.reason) {
          case VerifyCodeToConfirmPhoneProblemReason.CodeInvalid:
            code.addError({ rule: 'invalid' });
            break;

          case VerifyCodeToConfirmPhoneProblemReason.CodeNotSent:
            ConfirmPhoneBySmsSnackbars.codeNotSent();
            break;

          case VerifyCodeToConfirmPhoneProblemReason.ConfirmationsLimitExceeded:
            ConfirmPhoneBySmsSnackbars.confirmationsLimitExceeded();
            break;

          default:
            ConfirmPhoneBySmsSnackbars.internal();
            break;
        }
      }
    },
  });

  const handleAction = useCallback(
    (event: SmsCodeConfirmationEvent) => {
      switch (event.type) {
        case SmsCodeConfirmationEventType.CHANGE_CODE:
          code.onChange(event.code);
          break;

        case SmsCodeConfirmationEventType.RESEND_CODE:
          ConfirmPhoneBySmsSnackbars.hide();
          resend();
          break;

        case SmsCodeConfirmationEventType.VERIFY_CODE:
          ConfirmPhoneBySmsSnackbars.hide();
          submit();
          break;
      }
    },
    [code, resend, submit],
  );

  useFormEvents(form, {
    onSubmit: (input) => {
      verifyCode({
        variables: {
          input: { ...input, trackId },
        },
      });
    },
  });

  const pending = resendLoading || verifyLoading;

  return (
    <SmsCodeConfirmationForm
      pending={pending}
      number={number}
      expiryTimestamp={expiryTimestamp}
      onAction={handleAction}
      code={fields.code.value}
      invalid={fields.code.hasError()}
    />
  );
};
