/* eslint-disable @yandex-id/i18n/no-unwrapped-strings */
import { useRouter } from 'next/router';
import { useCallback, useEffect, useMemo, useRef, useState } from 'react';

import { useEnvContext } from '@client/shared/libs/env';
import { reachGoal } from '@client/shared/libs/metrika';
import { loadYandexPayScript } from '@client/shared/libs/yandex-pay';
import { ServerSideProps } from '@shared/types/next-bridge';

import { GibddBill, getBillsSum, getFee, getPrice } from '../gibdd';
import { createOrderBill } from '../gibdd/api/store';
import { cancelPayment, failPayment, waitPayment } from '../gibdd/model';
import { useBillsPayment } from '../gibdd/useBillsPayment';
import { Button, Order, Payment, PaymentData, PaymentEnv, YaPay } from './types';

type PaymentDataWithoutOrder = Omit<PaymentData, 'order'>;

const getInitialPaymentData = (
  config: ServerSideProps['yandexPayConfig'],
): PaymentDataWithoutOrder => {
  // @ts-expect-error  Нужно добавить тип в window
  const YaPay = window.YaPay as YaPay;

  // Сформировать данные платежа.
  return {
    env: config.env as PaymentEnv,
    version: 2,
    countryCode: YaPay.CountryCode.Ru,
    currencyCode: YaPay.CurrencyCode.Rub,
    merchant: {
      id: config.merchantId,
      name: config.merchantName,
    },
    paymentMethods: [
      {
        type: YaPay.PaymentMethodType.Card,
        gateway: config.gateway,
        gatewayMerchantId: config.gatewayId,
        allowedAuthMethods: [YaPay.AllowedAuthMethod.PanOnly],
        allowedCardNetworks: [
          YaPay.AllowedCardNetwork.Visa,
          YaPay.AllowedCardNetwork.Mastercard,
          YaPay.AllowedCardNetwork.Mir,
          YaPay.AllowedCardNetwork.Maestro,
          YaPay.AllowedCardNetwork.VisaElectron,
        ],
      },
    ],
    requiredFields: {
      billingContact: { email: true, name: true },
    },
  };
};

const getOrderByBills = (bills: GibddBill[], orderId: string): PaymentData['order'] => {
  const sum = getBillsSum(bills);
  const fee = getFee(bills);

  return {
    id: orderId,
    total: { label: 'Итого', amount: getPrice(sum + fee) },
    items: [
      { label: 'Сумма штрафа', amount: getPrice(sum) },
      { label: 'Комиссия', amount: getPrice(fee) },
    ],
  };
};

type PayRef = {
  init?: boolean;
  payment?: Payment;
  button?: Button;
  orderId?: string;
};

export const useYandexPay = () => {
  const router = useRouter();

  const { startPayment } = useBillsPayment();
  const { yandexPayConfig } = useEnvContext();

  const [yaPay, setYaPay] = useState<YaPay>();
  const [payment, setPayment] = useState<Payment>();
  const [button, setButton] = useState<Button>();
  const [buttonNode, setButtonNode] = useState<HTMLElement>();
  const [paymentOrder, setPaymentOrder] = useState<Order | null>(null);

  // Ref сделан для проброса данных в removeYandexPay и yaPay.PaymentEventType.Process
  const pay = useRef<PayRef>({});

  useEffect(() => {
    // @ts-expect-error Нужно добавить тип в window
    loadYandexPayScript(yandexPayConfig.scriptSrc, () => setYaPay(window.YaPay));
  }, [yandexPayConfig.scriptSrc]);

  const updatePay = useCallback((update: Partial<PayRef>) => {
    pay.current = { ...pay.current, ...update };
  }, []);

  useEffect(() => {
    const { init, payment } = pay.current;

    if (yaPay && paymentOrder && !init && !payment) {
      updatePay({ init: true });

      yaPay
        .createPayment({
          ...getInitialPaymentData(yandexPayConfig),
          order: paymentOrder,
        })
        .then((_payment) => {
          setPayment(_payment);
          updatePay({ payment: _payment });

          _payment.on(yaPay.PaymentEventType.Error, function onPaymentError() {
            if (_payment.sheet.order.id === pay.current.orderId) {
              failPayment();

              window.localStorage.removeItem('order-id');

              _payment.complete(yaPay.CompleteReason.Error, 'error');
            }
          });

          _payment.on(yaPay.PaymentEventType.Abort, function onPaymentAbort() {
            if (_payment.sheet.order.id === pay.current.orderId) {
              cancelPayment();

              window.localStorage.removeItem('order-id');

              _payment.complete(yaPay.CompleteReason.Close);
            }
          });

          _payment.on(yaPay.PaymentEventType.Process, function onPaymentProcess(event) {
            if (_payment.sheet.order.id === pay.current.orderId) {
              startPayment(
                pay.current.orderId,
                event.token,
                event.billingContact?.name || 'Иванов Иван',
              );

              window.localStorage.removeItem('order-id');

              _payment.complete(yaPay.CompleteReason.Success);
            }
          });
        });
    }
  }, [yaPay, paymentOrder, updatePay, startPayment, yandexPayConfig]);

  useEffect(() => {
    const { payment, orderId } = pay.current;

    if (yaPay && payment && paymentOrder && paymentOrder.id !== orderId) {
      payment.update({ order: paymentOrder });
    }
  }, [yaPay, paymentOrder]);

  useEffect(() => {
    if (yaPay && buttonNode && payment) {
      const button = payment.createButton({
        type: yaPay.ButtonType.Pay,
        theme: yaPay.ButtonTheme.Yellow,
        width: yaPay.ButtonWidth.Max,
      });

      button.mount(buttonNode);

      button.on(yaPay.ButtonEventType.Click, function onPaymentButtonClick() {
        reachGoal('click_pay');

        const {
          sheet: {
            order: { id: orderId },
          },
        } = payment;

        try {
          window.localStorage.setItem('order-id', orderId);
        } catch (_error) {
          // TODO: сделать fallback, если отключён localStorage.
        }

        payment.checkout();
        updatePay({ orderId });
        waitPayment();
      });

      setButton(button);
      updatePay({ button });
    }
  }, [yaPay, buttonNode, updatePay, payment]);

  const isYandexPayReady = useMemo(() => Boolean(button), [button]);

  const updateYandexPay = useCallback(
    (node: HTMLElement, bills: GibddBill[]) => {
      setButtonNode(node);

      const localOrderId = window.localStorage.getItem('order-id');

      // Создаём объект Pay, т.к. вернулись с редиректа и транзакция в процессе
      if (localOrderId && router.query.__YP__) {
        setPaymentOrder(getOrderByBills([], localOrderId));
        updatePay({ orderId: localOrderId });
        waitPayment();
      } else if (bills.length) {
        const ids = bills.map((bill) => bill.bill_id);

        createOrderBill({ ids }).then(
          (data) => data && setPaymentOrder(getOrderByBills(bills, data.order_id)),
        );
      }
    },
    [updatePay, router.query.__YP__],
  );

  const removeYandexPay = useCallback(() => {
    const { payment, button } = pay.current;

    if (button) {
      button.unmount();
    }

    if (payment) {
      payment.destroy();
    }
  }, []);

  return {
    isYandexPayReady,
    updateYandexPay,
    removeYandexPay,
  };
};
